import attr
from django.utils.functional import cached_property

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import Source
from infra.cauth.server.public.api.helpers import get_access_uids, get_responsible_uids, get_users
from infra.cauth.server.public.api.views.base import BaseSourceView
from infra.cauth.server.public.constants import SOURCE_NAME


class Color:
    colors = {
        'green': r'\e[0;32m',
        'yellow': r'\e[0;33m',
        'red': r'\e[0;31m',
        'cyan': r'\e[0;36m',
        'bold': r'\e[1;39m',
        'under': r'\e[4m',
        'normal': r'\e[0;39m',
    }

    def __init__(self, use_color):
        self.use_color = use_color

    def __getattr__(self, item):
        if not self.use_color:
            return ''

        return self.colors.get(item, '')


@attr.s(repr=False)
class UserInfo:
    login = attr.ib()
    has_ssh = attr.ib(default=True, cmp=False)
    is_root = attr.ib(default=False, cmp=False)
    sources = attr.ib(default=attr.Factory(set), cmp=False)

    def __repr__(self):
        return '{}: {}'.format(self.login, ','.join(sorted(self.sources)))


class InfoController:
    def __init__(self, server, sources, can_use_access):
        self.server = server
        self.sources = sources
        self.can_use_access = can_use_access

    @cached_property
    def idm_access(self):
        return get_access_uids(self.server)

    @cached_property
    def responsibles(self):
        return get_responsible_uids(self.server)

    @cached_property
    def logins_map(self):
        all_uids = self.idm_access.keys() | self.responsibles.keys()
        user_query = get_users(all_uids)
        return {user.uid: user.login for user in user_query}

    @cached_property
    def sources_idm_access(self):
        return get_access_uids(self.server, self.sources, self.can_use_access)

    @cached_property
    def sources_responsibles(self):
        return get_responsible_uids(self.server, self.sources)

    @cached_property
    def sources_map(self):
        return {source.id: source.name for source in Session.query(Source)}

    @cached_property
    def ssh_uids(self):
        return self.sources_idm_access.keys() | self.sources_responsibles.keys()

    def get_has_ssh(self, uid):
        return uid in self.ssh_uids

    def get_root_status(self, uid):
        return self.sources_idm_access.get(uid, False) or uid in self.sources_responsibles

    def get_sources_names(self, uid):
        sources = self.responsibles.get(uid) or set()
        sources = {self.sources_map[id_] for id_ in sources}
        if uid in self.idm_access:
            sources.add(SOURCE_NAME.IDM)
        return sources

    def create_user_info(self, uid):
        return UserInfo(
            login=self.logins_map[uid],
            has_ssh=self.get_has_ssh(uid),
            is_root=self.get_root_status(uid),
            sources=self.get_sources_names(uid),
        )

    @cached_property
    def users_info(self):
        return [self.create_user_info(uid) for uid in self.logins_map]

    @cached_property
    def ssh_users(self):
        return [user for user in self.users_info if user.has_ssh and not user.is_root]

    @cached_property
    def root_users(self):
        return [user for user in self.users_info if user.has_ssh and user.is_root]

    @cached_property
    def disabled_users(self):
        return [user for user in self.users_info if not user.has_ssh]


class SourcesInfoView(BaseSourceView):
    template_name = 'api/info.txt'

    default_context = {
        'ip': None,
        'fqdn': None,
        'sources': [],
        'sudoers': [],
        'ssh_users': [],
        'root_users': [],
        'disabled_users': [],

        'color': Color(use_color=False),
    }

    @cached_property
    def use_color(self):
        return bool(self.request.GET.get('color', ''))

    def get_prepared_sources(self):
        if self.sources is None:
            return None

        if not self.sources:
            return 'empty (invalid behavior)'

        return ','.join(sorted([source.name for source in self.sources]))

    def make_context(self):
        controller = InfoController(self.remote_server, self.sources, self.can_use_access)

        return {
            'ip': self.effective_remote_addr,
            'fqdn': self.effective_remote_fqdn,
            'sources': self.get_prepared_sources(),
            'sudoers': self.get_sudoers(),
            'ssh_users': sorted(controller.ssh_users),
            'root_users': sorted(controller.root_users),
            'disabled_users': sorted(controller.disabled_users),

            'color': Color(use_color=self.use_color),
        }
