import collections
import json

from django.utils.functional import cached_property

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.constants import FLOW_TYPE, KEYS_INFO_ATTR
from infra.cauth.server.common.models import (
    User,
    Group,
    UserGroupRelation,
    ServerTrustedSourceRelation,
    Server,
)
from infra.cauth.server.public.api.helpers import get_keys, get_root_uids, get_access_uids, get_responsible_uids
from infra.cauth.server.public.api.views.base import BaseView, BaseSourceView
from infra.cauth.server.public.master_api import MasterApiClient

Key = collections.namedtuple('Key', 'login body')


def sanitize_string(body):
    return body.replace('\n', '\\n').replace('\t', '\\t').replace('\b', '\\b')


class UserkeysView(BaseSourceView):
    template_name = 'api/userkeys.txt'

    default_context = {
        'user_keys': [],
        'admin_keys': [],
    }

    def get_sources_for_update(self):
        if self.sources is None:
            return None

        db_sources = (
            Session.query(ServerTrustedSourceRelation.source_id)
            .join(Server)
            .filter(Server.id == self.remote_server.id)
        )
        db_sources_ids = {source_id for (source_id,) in db_sources}
        trusted_sources_ids = {source.id for source in self.sources}

        if db_sources_ids == trusted_sources_ids:
            return None

        return self.sources

    def get_client_version_for_update(self):
        if self.client_version is None or self.remote_server.client_version == self.client_version:
            return None

        return self.client_version

    def update_server(self):
        if self.q_param is not None or self.remote_server is None or self.client_version is None:
            return
        if self.remote_server.flow != FLOW_TYPE.CLASSIC:
            return

        sources = self.get_sources_for_update()
        client_version = self.get_client_version_for_update()

        if sources is None and client_version is None:
            return

        MasterApiClient.update_server(self.remote_server, sources, client_version)

    def make_context(self):
        access_uids = get_access_uids(self.remote_server, self.sources, self.can_use_access)

        for uid in get_responsible_uids(self.remote_server, self.sources).keys():
            access_uids[uid] = True

        keys_query = get_keys(list(access_uids.keys()), entities=[User.uid, User.login])

        user_keys = []
        admin_keys = []
        for key, uid, login in keys_query:
            key = Key(login, sanitize_string(key))

            if access_uids[uid]:
                admin_keys.append(key)
            else:
                user_keys.append(key)

        self.update_server()

        return {
            'user_keys': user_keys,
            'admin_keys': admin_keys,
        }


class AdminkeysView(BaseSourceView):
    template_name = 'api/adminkeys.txt'

    default_context = {
        'admin_keys': [],
    }

    def make_context(self):
        uids = get_root_uids(self.remote_server, self.sources, self.can_use_access)

        admin_keys = [Key(None, sanitize_string(key)) for key, in get_keys(uids)]
        return {
            'admin_keys': admin_keys,
        }


class UnitkeysView(BaseView):
    template_name = 'api/unitkeys.txt'

    @cached_property
    def unit_name(self):
        return self.kwargs.get('name', '')

    @cached_property
    def unit(self):
        name = self.unit_name

        user = User.query.filter_by(login=name).first()
        if user:
            return user

        group = Group.query.filter_by(name=name).first()
        if group:
            return group

    def get_context_data(self):
        unit = self.unit

        if unit:
            if isinstance(unit, User):
                uids = {unit.uid}
            else:
                uids_query = (
                    Session.query(UserGroupRelation.uid)
                    .filter(UserGroupRelation.gid == unit.gid)
                )
                uids = {uid for uid, in uids_query}
            keys_query = get_keys(uids, entities=[User.login])
            keys = [Key(login, sanitize_string(key)) for key, login in keys_query]
        else:
            keys = []

        return {
            'name': self.unit_name,
            'keys': keys,
        }


class KeysInfoView(BaseSourceView):
    template_name = 'api/keysinfo.txt'
    with_response_wrapper = False
    content_type = 'application/json; charset=utf-8'

    default_context = {
        "body": "{}"
    }

    def make_context(self):
        data = {
            KEYS_INFO_ATTR.KEY_SOURCES: self.remote_server.get_keys_info(KEYS_INFO_ATTR.KEY_SOURCES).split(','),
            KEYS_INFO_ATTR.SECURE_CA_LIST_URL: self.remote_server.get_keys_info(KEYS_INFO_ATTR.SECURE_CA_LIST_URL),
            KEYS_INFO_ATTR.INSECURE_CA_LIST_URL: self.remote_server.get_keys_info(KEYS_INFO_ATTR.INSECURE_CA_LIST_URL),
            KEYS_INFO_ATTR.KRL_URL: self.remote_server.get_keys_info(KEYS_INFO_ATTR.KRL_URL),
            KEYS_INFO_ATTR.SUDO_CA_LIST_URL: self.remote_server.get_keys_info(KEYS_INFO_ATTR.SUDO_CA_LIST_URL),
        }

        return {
            "body": json.dumps(data),
        }
