import collections
import itertools

from django.core.cache import cache

from infra.cauth.server.public.api.helpers import (
    get_posix_passwd_new,
    get_posix_groups_new,
    get_root_uids,
    get_access_uids,
    get_responsible_uids,
    get_users,
    get_users_gids,
)
from infra.cauth.server.public.api.views.base import BaseView, BaseSourceView

GroupItem = collections.namedtuple('GroupItem', 'name gid users')
UserItem = collections.namedtuple('UserItem', 'login uid gid first_name last_name home shell')


class CachedBodyMixin:
    def render_body(self):
        entry = cache.get(self.cache_key)

        if entry:
            timestamp, body = entry
            return body

        return super(CachedBodyMixin, self).render_body()


class AllGroupsView(CachedBodyMixin, BaseView):
    template_name = 'api/all_groups.txt'
    cache_key = 'full_body:/group/'

    @classmethod
    def item_from_tuple(cls, values):
        return GroupItem(*values)

    def get_context_data(self):
        return {
            'items': [GroupItem(*values) for values in get_posix_groups_new()],
        }


class AllPasswdView(CachedBodyMixin, BaseView):
    template_name = 'api/all_passwd.txt'
    cache_key = 'full_body:/passwd/'

    @classmethod
    def item_from_tuple(cls, values):
        return UserItem(*values)

    def get_context_data(self):
        return {
            'items': list(map(self.item_from_tuple, get_posix_passwd_new())),
        }


class AdminsGroupView(BaseSourceView):
    template_name = 'api/admins_group.txt'

    default_context = {
        'logins': [],
    }

    def make_context(self):
        uids = get_root_uids(self.remote_server, self.sources, self.can_use_access)
        admins = get_users(uids)

        return {
            'logins': [user.login for user in admins],
        }


class UsersGroupView(BaseSourceView):
    template_name = 'api/users_group.txt'

    default_context = {
        'items': [],
    }

    @property
    def return_all_users_in_group(self):
        return 'all_users' in self.request.GET

    def get_groups(self):
        access_uids = get_access_uids(self.remote_server, self.sources, self.can_use_access)
        responsible_uids = get_responsible_uids(self.remote_server, self.sources)

        serverusers_uids = {uid for uid in itertools.chain(access_uids, responsible_uids)}
        if not serverusers_uids:
            return []

        if self.return_all_users_in_group:
            filters = {'gids': get_users_gids(serverusers_uids)}
        else:
            filters = {'uids': serverusers_uids}
        return get_posix_groups_new(**filters)

    def make_context(self):
        return {
            'items': [GroupItem(*values) for values in self.get_groups()],
        }


class AdminsPasswdView(BaseSourceView):
    template_name = 'api/admins_passwd.txt'

    default_context = {
        'admins': [],
    }

    def make_context(self):
        uids = get_root_uids(self.remote_server, self.sources, self.can_use_access)
        admins = get_users(uids)

        return {
            'admins': admins,
        }


class UsersPasswdView(BaseSourceView):
    template_name = 'api/users_passwd.txt'

    default_context = {
        'users': [],
        'admins': [],
    }

    def make_context(self):
        access_uids = get_access_uids(self.remote_server, self.sources, self.can_use_access)

        for uid in get_responsible_uids(self.remote_server, self.sources).keys():
            access_uids[uid] = True

        query = get_users(list(access_uids.keys()))
        admins = []
        users = []
        for user in query:
            if access_uids[user.uid]:
                admins.append(user)
            else:
                users.append(user)

        return {
            'users': users,
            'admins': admins,
        }
