import datetime
import pprint
import random
import threading
import time
import logging
import os
import signal

import tornado.gen
import tornado.ioloop

from concurrent.futures import ProcessPoolExecutor
from tornado.gen import coroutine


class AutoUpdatingContext(object):
    def __init__(self):
        self._executor = None
        self._context = {}
        self._updaters = []

    def __getitem__(self, item):
        return self._context[item]

    def submit(self, name, update_function, default=None, args=None, callback=None, delay=3.0, timeout=30.0):
        self._context[name] = {
            'timestamp': datetime.datetime.min,
            'data': default,
        }
        self._updaters.append(
            _Updater(name, args, update_function, tornado.ioloop.IOLoop.instance(),
                     self._context, delay, timeout, callback)
        )

    def start(self):
        self._executor = ProcessPoolExecutor(max_workers=len(self._context))
        for updater in self._updaters:
            updater.start(self._executor)

    def stop(self):
        _logger.info('shutting down')
        self._executor.shutdown(wait=False)


class _Updater(object):
    def __init__(self, name, args, update_function, ioloop, context, delay, timeout, callback=None):
        self.update_function = update_function
        self.ioloop = ioloop
        self.context = context
        self.name = name
        self.args = args or []
        self.callback = callback
        self.delay = delay
        self.timeout = timeout
        self.executor = None

    def start(self, executor):
        self.executor = executor
        self.ioloop.spawn_callback(self._run_loop)

    def _apply_result(self, result):
        # noinspection PyBroadException
        try:
            self.context[self.name].update(result)
            if self.callback is not None:
                self.callback(result)
        except:
            _logger.exception('[%s] callback failed', self.name)

    @coroutine
    def _run_loop(self):
        none = datetime.datetime.min
        while True:
            if any(self.context.get(arg, {}).get('timestamp', none) == none for arg in self.args):
                _logger.info('[%s] args not ready', self.name)
                yield tornado.gen.sleep(1.0)
                continue

            kwargs = {arg: self.context.get(arg, {}).get('data', None) for arg in self.args}
            kwargs['timestamp'] = self.context[self.name]['timestamp']

            _logger.info('[%s] update scheduled from %s', self.name, kwargs['timestamp'])
            _logger.info('executor has %s processes', len(self.executor._processes))

            future = self.executor.submit(
                _exec_with_timeout,
                self.update_function,
                kwargs,
                timeout=self.timeout
            )

            # noinspection PyBroadException
            try:
                result = yield future
            except _Timeout as e:
                for p in list(self.executor._processes):
                    if p.pid == e.pid:
                        _logger.warn('[%s] update timed out, killed pid %s', self.name, e.pid)
                        os.kill(e.pid, signal.SIGKILL)
                        self.executor._processes.discard(p)
            except:
                _logger.exception('[%s] update failed', self.name)
            else:
                self._apply_result(result)
                _logger.info('[%s] update succeeded', self.name)

            yield tornado.gen.sleep(self.delay)


def _exec_with_timeout(function, kwargs, timeout):
    result = []
    error = []

    def _exec_and_capture_result():
        try:
            result.append(function(**kwargs))
        except Exception as e:
            error.append(e)

    guard = threading.Thread(target=_exec_and_capture_result)
    guard.daemon = True
    guard.start()
    guard.join(timeout=timeout)

    if guard.is_alive():
        raise _Timeout
    else:
        if error:
            raise error[0]
        return result[0]


class _Timeout(RuntimeError):
    def __init__(self):
        import os
        self.pid = os.getpid()


_logger = logging.getLogger('pool')


def test_simple():
    scheduler = AutoUpdatingContext()

    def dump(*args, **kwargs):
        pprint.pprint(scheduler._context)

    scheduler.submit('fn1', fn1, args=['fn2'], delay=0.1, timeout=1, callback=dump)
    scheduler.submit('fn2', fn2, args=['fn1'], delay=0.1, timeout=1)
    scheduler.start()

    scheduler._context['fn1'].update({'timestamp': datetime.datetime.now(), 'data': 0})  # seed

    tornado.ioloop.IOLoop.instance().start()


def fn1(timestamp, fn2):
    time.sleep(random.uniform(0, 2))
    if random.uniform(0, 1) > 0.75:
        raise Exception('random exception')

    return {'timestamp': datetime.datetime.now(), 'data': fn2 + 1}


def fn2(timestamp, fn1):
    time.sleep(random.uniform(0, 2))
    if random.uniform(0, 1) > 0.75:
        raise Exception('random exception')

    return {'timestamp': datetime.datetime.now(), 'data': fn1 + 1}


if __name__ == '__main__':
    #logging.basicConfig(level=logging.DEBUG)
    test_simple()
