import datetime
import json
import logging
import threading

from tornado import gen
from tornado.web import RequestHandler

from jinja2 import Environment, FileSystemLoader

from pool import pool
from utils import genuuid
from yasm import stats_manager


class JSONEncoder(json.JSONEncoder):
    def default(self, obj):
        from libraries.online_state.state_obj import IStateV3, ilongname_key_map

        if isinstance(obj, set):
            return sorted(obj)

        if isinstance(obj, IStateV3):
            return {key: obj[ilongname_key_map[key]] for key in ilongname_key_map}

        return json.JSONEncoder.default(self, obj)


class BaseHandler(RequestHandler):
    def __init__(self, application, request, **kwargs):
        super(BaseHandler, self).__init__(application, request, **kwargs)

        self.reqid = genuuid()[:8]
        self.log = getattr(application, 'log', logging.getLogger())

    def render_template(self, template_name, **kwargs):
        try:
            template_dirs = self.settings.get('template_path', [])
            env = Environment(loader=FileSystemLoader(template_dirs))
            env.filters['reverse_url'] = self.reverse_url
            return env.get_template(template_name).render(kwargs)
        except:
            logging.exception('Rendering exception')
            raise

    def is_json(self):
        json = self.get_argument('json', default='', strip=True)
        return json.lower() in ('da', 'yes', 'true', '1')

    def render2(self, template_name, **kwargs):
        if self.is_json():
            self.set_header('Content-Type', 'application/json')
            try:
                content = json_encode_pretty(kwargs)
            except TypeError:
                if 'details' in kwargs:
                    # generator object in kwargs['details']
                    kwargs.update({'details': list(kwargs['details'])})
                    content = json_encode_pretty(kwargs)
                else:
                    raise
        else:
            kwargs.setdefault('show_footer', True)
            kwargs.update({
                'settings': self.settings,
                'STATIC_URL': self.settings.get('static_url_prefix', '/static/'),
                'request': self.request,
                'response_code': self.get_status(),
                'xsrf_token': self.xsrf_token,
                'xsrf_form_html': self.xsrf_form_html,
                'backend_hostname': self.settings.get('backend_hostname', 'hostname'),
            })
            content = self.render_template(template_name, **kwargs)

        self.write(content)

    def post(self, **kwargs):
        self.check_xsrf_cookie()

    def data_received(self, chunk):
        pass

    def prepare(self):
        self.log.info('start %s %s [%s]', self.request.method, self.request.uri, self.reqid)

    def on_finish(self):
        name = type(self).__name__.lower()
        stats_manager.incrementCounter("request_count_%s" % name)
        stats_manager.addSample("request_time_%s" % name, self.request.request_time())


class BaseAsyncHandler(BaseHandler):
    json_response = False
    template_name = None
    timeout = 1
    vip = False

    def set_extra_headers(self, path):
        self.set_header('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0')

    def _process_request(self, fn):
        if self.template_name is None and not self.json_response:
            raise Exception('template_name is not set')

        def _setup_and_run():
            try:
                threading.current_thread().name = self.reqid
                return fn()
            except:
                logging.exception('Unhandled exception in thread pool')
                raise

        target_pool = pool() if not self.vip else self.application.pool

        future = gen.with_timeout(
            datetime.timedelta(seconds=self.timeout),
            target_pool.submit(_setup_and_run)
        )

        try:
            data = yield future

            if data is not None:
                if self.json_response:
                    self.set_header('Content-Type', 'application/json')
                    self.write(json.dumps(data, indent=4, cls=JSONEncoder).replace('</', '<\\/'))
                else:
                    self.render2(self.template_name, **data)

        except gen.TimeoutError:
            self.set_status(500)

    @gen.coroutine
    def get(self, *args, **kwargs):
        return self._process_request(self.get_data)

    def get_data(self):
        self.set_status(405)

    @gen.coroutine
    def post(self, *args, **kwargs):
        return self._process_request(self.post_data)

    def post_data(self):
        self.set_status(405)

    @gen.coroutine
    def delete(self, *args, **kwargs):
        return self._process_request(self.delete_data)

    def delete_data(self):
        self.set_status(405)


def json_encode_pretty(value):
    return json.dumps(value, indent=4, cls=JSONEncoder).replace("</", "<\\/")
