from collections import defaultdict
import time

# noinspection PyUnresolvedReferences
import sepelib.yandex.iss

from libraries.cms import Registry
# noinspection PyUnresolvedReferences
from utils import singleton, shortname
import hqcms


def all_conf(prefix=None):
    return _cms().all_conf(prefix)


def conf_instances(conf, conf_mtime, itag=None):
    return _cms().conf_instances(conf, conf_mtime, itag)


def conf_shards(conf, conf_mtime, itag=None):
    return _cms().conf_shards(conf, conf_mtime, itag)


def get_shards(interested_shards, conf='HEAD'):
    return _cms().get_shards(interested_shards, conf)


def conf_itags(conf, conf_mtime):
    return _cms().conf_itags(conf, conf_mtime)


def conf_topology(conf):
    return _cms().conf_topology(conf)


def suggest(term):
    return _cms().suggest(term)


def suggest_no_mainconf(term):
    return _cms().suggest(term)[0]


def update_cache():
    _cms().update_cache()


def get_full_instances(conf, instances_filter):
    return _cms().get_full_instances(conf, instances_filter)


class _ConfCache(object):
    def __init__(self, cms, hq_cache):
        self._cms = cms
        self._cms_cache = {}
        self._hq_cache = hq_cache

    def list(self, prefix):
        result = self._cms_cache.copy()

        if prefix:
            for c in self._cms_cache:
                if not c.startswith(prefix):
                    result.pop(c)

        result.update(self._hq_cache.list_configurations(prefix))
        return result

    def update(self):
        self._cms_cache = {
            conf['name']: conf['mtime'] for conf in self._cms.listConf()
        }


class _InstancesCache(object):
    def __init__(self, cms, hq_cache):
        self._cms = cms
        self._cache = defaultdict(dict)
        self._hq_cache = hq_cache

    def conf_instances(self, conf, mtime, itag):
        self._cache[conf].setdefault(itag, {'mtime': 0, 'instances': {}})

        if self._cache[conf][itag]['mtime'] < mtime:
            instances = self._find_hq_instances(conf, itag)
            if not instances:
                instances = self._request_cms_instances(conf, itag)

            self._cache[conf][itag]['instances'] = instances
            self._cache[conf][itag]['mtime'] = mtime

        return self._cache[conf][itag]['instances']

    def _request_cms_instances(self, conf, itag):
        filters = {'conf': conf}
        if itag:
            filters['instanceTagName'] = itag

        instances = defaultdict(dict)
        for i in self._cms.listSearchInstances(filters):
            instances[i['host']]['{}:{}'.format(i['host'], i['port'])] = i['shard']

        return dict(instances)

    def _find_hq_instances(self, conf, itag):
        # format: {'sas1-9271': {'sas1-9271:17880': 'none'},
        #          'sas1-9402': {'sas1-9402:17880': 'none'},
        #          'sas1-9618': {'sas1-9618:17880': 'none'},
        #          'sas1-9619': {'sas1-9619:17880': 'none'}}
        result = defaultdict(dict)
        for instance in self._hq_cache.list_instances(conf, itag):
            short_hostname = instance[0].split('.')[0]
            short_instance = '{}:{}'.format(short_hostname, str(instance[1]))
            result[short_hostname][short_instance] = 'none'

        return dict(result)


class _CMS(object):
    def __init__(self):
        self._cms = Registry.nativeInterface()
        self._iss = sepelib.yandex.iss.IssApiWrapper(request_timeout=5, max_tries=3)

        self._hq_cache = hqcms.HqCache()
        self._conf_cache = _ConfCache(self._cms, self._hq_cache)
        self._instances_cache = _InstancesCache(self._cms, self._hq_cache)

        # TODO
        self._itags_cache = {}
        self._topology_cache = {}

        # no hq analogue
        self._shards_cache = {}
        self._shards_update_interval = 30 * 60

    def update_cache(self):
        self._conf_cache.update()
        self._hq_cache.update()

    def all_conf(self, prefix=None):
        return self._conf_cache.list(prefix)

    def conf_instances(self, conf, mtime, itag=None):
        return self._instances_cache.conf_instances(conf, mtime, itag)

    def conf_shards(self, conf, conf_mtime, itag=None):
        self._shards_cache.setdefault(conf, {})
        self._shards_cache[conf].setdefault(itag, {'mtime': 0, 'shards': {}})

        if time.time() - self._shards_cache[conf][itag]['mtime'] > self._shards_update_interval:
            shards = set()
            for h, j in self.conf_instances(conf, conf_mtime, itag).iteritems():
                for i, s in j.iteritems():
                    shards.add(s)

            new_shards = {}

            if len(shards) > 0:
                all_shards_tags = self._cms.listAllShardsTags({'conf': conf, 'shards': list(shards)})
                for shard in all_shards_tags:
                    sn = shard['name']
                    if 'Tier' in sn or 'Maps' in sn or 'geoshard' in sn or 'Jud0' in sn or 'Kz0' in sn\
                            or 'Ita0' in sn or 'Tur-1' in sn or 'Div' in sn or 'OxygenExp' in sn or 'Eng0' in sn\
                            or 'Rrg0' in sn or 'Tur0' in sn or 'Rus-1' in sn:
                        sn = sn.replace('Tier', '')
                        new_shards[shard['shard']] = dict(tier=sn, cms_mtime=-1)

                all_shards = self._cms.getShards(list(shards))
                for shard in all_shards:
                    if shard['name'] in new_shards:
                        new_shards[shard['name']]['cms_mtime'] = shard['mtime']

            self._shards_cache[conf][itag]['mtime'] = time.time()
            self._shards_cache[conf][itag]['shards'] = new_shards

        return self._shards_cache[conf][itag]['shards']

    def get_shards(self, interested_shards, conf='HEAD'):
        q = {}
        r = {}

        if conf in self.all_conf():
            shards = self.conf_shards(conf, self.all_conf()[conf])
            for k, v in shards.iteritems():
                if k in interested_shards:
                    q[k] = v['tier']
                    r[k] = v['cms_mtime']

        return r, q

    def conf_itags(self, conf, conf_mtime):
        self._itags_cache.setdefault(conf, {'mtime': 0, 'itags': []})

        if self._itags_cache[conf]['mtime'] < conf_mtime:
            tags = self._cms.listAllSearchInstanceTags({'conf': conf})
            self._itags_cache[conf]['mtime'] = conf_mtime
            self._itags_cache[conf]['itags'] = tags

        return self._itags_cache[conf]['itags']

    def conf_topology(self, conf):
        if conf in self._topology_cache:
            return self._topology_cache[conf]

        all_confs = self.all_conf()

        if conf not in all_confs:
            return ''

        c = 0
        ret = ''
        for i in self.conf_itags(conf, all_confs[conf]):
            if i['name'].startswith('a_topology_version'):
                c += 1
                ret = i['name']

        if c == 1:
            self._topology_cache[conf] = ret
            return ret

        return ''

    def suggest(self, term):
        def conf_suggest(pr):
            return sorted(self.all_conf(pr).keys())

        def itag_suggest(pr, conf):
            itags = set()
            all_confs = self.all_conf()
            if conf not in all_confs.keys():
                return []
            all_itags = set()
            for i in self.conf_itags(conf, all_confs[conf]):
                all_itags.add(i['name'])
            if pr != '' and len(all_itags):
                for itag in all_itags:
                    if itag.startswith(pr):
                        itags.add(itag)
                return sorted(itags)
            else:
                return sorted(all_itags)

        complete_with = []
        main_conf = 'HEAD'
        prefix = term.rsplit(' ', 1)[0]
        term = term.rsplit(' ', 1)[-1]
        if prefix == term:
            prefix = ''
        else:
            prefix += ' '
            conf_count = 0
            for p in prefix.split(' '):
                if p.startswith('C@'):
                    main_conf = p.split('C@', 1)[-1]
                    conf_count += 1
            if conf_count != 1:
                main_conf = 'HEAD'

        if term.startswith('C@'):
            for c in conf_suggest(term.split('C@', 1)[-1]):
                complete_with.append(prefix + 'C@' + c)
                if len(complete_with) == 1:
                    main_conf = term.split('C@', 1)[-1]
        elif term.startswith('I@'):
            for c in itag_suggest(term.split('I@', 1)[-1], main_conf):
                complete_with.append(prefix + 'I@' + c)

        if main_conf not in self.all_conf().keys():
            main_conf = 'HEAD'

        return complete_with, main_conf

    def get_full_instances(self, conf, instances_filter):
        system, conf = self._detect_configuration_source(conf)

        if system == 'iss':
            return self._get_iss_instances(conf, instances_filter)
        elif system == 'cms':
            return self._get_cms_instances(conf, instances_filter)

        raise RuntimeError('Configuration from unknown source %s', conf)

    def _detect_configuration_source(self, conf):
        if '#' in conf:  # otherwise iss raises an exception
            iss_conf = self._iss.get_configuration_by_id(conf)
            if iss_conf and iss_conf.applicable:
                return 'iss', conf

        try:
            heuristic_iss_conf = conf.split('-')[0] + '#' + conf
            iss_conf = self._iss.get_configuration_by_id(heuristic_iss_conf)
            if iss_conf and iss_conf.applicable:
                return 'iss', heuristic_iss_conf

        except sepelib.yandex.iss.IssValidationError:
            pass

        return 'cms', conf

    def _get_cms_instances(self, conf, instances_filter):
        cms_instances = self._cms.listSearchInstances({'conf': conf})
        shard_names = [instance['shard'] for instance in cms_instances]

        cms_shards = self._cms.getShards(shard_names)
        shards = {shard['name']: shard for shard in cms_shards}

        shard_tags = defaultdict(list)
        for shard_tag in self._cms.listAllShardsTags({'conf': conf, 'shards': shard_names}):
            shard_tags[shard_tag['shard']].append(shard_tag['name'])

        instances = {}
        for instance in cms_instances:
            name = '{}:{}'.format(instance['host'], instance['port'])
            if instances_filter is None or name in instances_filter:
                instances[name] = {
                    'tags': shard_tags[instance['shard']],
                    'shards': {instance['shard']: shards.get(instance['shard'], {}).get('mtime', -1)},
                }

        return instances

    def _get_iss_instances(self, conf, instances_filter):
        iss_instances = self._iss.get_configuration_instances_with_resources(conf)

        instances = {}
        for instance in iss_instances:
            name = '{}:{}'.format(shortname(instance.host), instance.service)
            if instances_filter is None or name in instances_filter:
                instances[name] = {
                    'tags': instance.properties['properties/tags'].split(' '),
                    'shards': {shard: 0 for shard in instance.shards.iterkeys()},
                }

        return instances


@singleton
def _cms():
    return _CMS()
