# coding=utf-8
import datetime
import json
import collections
import urllib
import pytz
import re
from itertools import chain

from tornado import gen
from tornado.escape import json_encode
from copy import deepcopy
from tornado.web import RequestHandler
from libraries.hardware import hosts_for_yasm
from libraries.waitinfo import waitinfo

from panels import PANELS

from analyzer import analyzer
from blinov import resolve
from states import states
from suspension_api import acquire, get_status, release

import cms

from base_handler import BaseHandler, BaseAsyncHandler
from libraries import topology
from yasm import stats_manager

from states import get_coll

from mongo_params import ALL_HEARTBEAT_C_MONGODB, HEARTBEAT_MONGODB


class SearchHandler(BaseAsyncHandler):
    template_name = 'search.html'
    timeout = 30

    def get_data(self):
        term = self.get_argument('term', '', True)
        if term == 'C@production_sas_mmeta' and self.get_argument('signal', default=False):
            return {
                'term': term,
                'error': False,
                'error_description': None
            }

        page = self.get_argument('page', 'kernels', True)

        complete_with, main_conf, hosts, instances, single_conf, configs = self.extend_request(term)

        data = {
            'term': term,
            'page': page,
            'hosts': len(instances) or len(hosts),
            'main_conf': main_conf,
            'hosts_count': len(hosts),
            'error': False,
            'error_description': None
        }

        if not single_conf:
            confs = {}

            for c in configs:
                if not states().active_configurations():
                    raise RuntimeError('Resolver stalled')

                _, instances = resolve('C@' + c)
                if len(instances):
                    state = analyzer().configuration_simple(c, instances)
                    state['hosts_active_perc'] = \
                        state['hosts_active_count'] / float(state['hosts_active_count'] + state['hosts_inactive_count'])
                    if state['hosts_active_count'] > 0:
                        confs[c] = state
                else:
                    raise RuntimeError('Unable to resolve {}'.format(c))

            data['confs_keys'] = sorted(confs, key=lambda x: confs[x]['hosts_active_perc'], reverse=True)
            data['confs'] = confs

        elif len(instances) == 0 and len(hosts) == 0 and term:
            self.set_status(404)
            data['error'] = True
            data['error_description'] = 'No instances found'

        else:
            data['single_configuration'] = True

            if len(hosts) > 0:
                if page == 'kernels':
                    data['versions'] = analyzer().versions(hosts)
                    data['versions2'] = analyzer().versions2(hosts)
                    data['versions3'] = analyzer().versions3(hosts)
                    data['versions4'] = analyzer().versions4(hosts)
                elif page == 'packages':
                    data['packages'] = analyzer().packages(hosts)
                elif page == 'skynet':
                    data['skyinfo'] = analyzer().skyinfo(hosts)
                    data['mon'] = analyzer().monitoring(hosts)
                elif page == 'configurations':
                    data['confs'] = analyzer().configurations_by_hosts(hosts)

            if len(instances) > 0:
                instances_filter = frozenset(chain.from_iterable(inst.keys() for inst in instances.itervalues()))
                if page == 'deploy':
                    data.update(analyzer().deploy(main_conf, instances_filter))
                elif page == 'details':
                    get_all = self.get_argument('get_all', '').lower() in ('true', 'yes', 'da', '1')
                    data.update({'details': analyzer().details(main_conf, instances_filter, get_all)})
                elif page == 'instances':
                    data['instances'] = analyzer().instances_complex(main_conf, instances)
                elif page == 'configurations':
                    data['confs'] = analyzer().configurations_by_hosts(hosts)
                elif page == 'shardtool':
                    get_all = self.get_argument('get_all', '').lower() in ('true', 'yes', 'da', '1')
                    data.update({'shardtool': analyzer().shardtool(main_conf, instances_filter, get_all)})
        return data

    def extend_request(self, term):
        complete_with, main_conf = cms.suggest(term)
        hosts, instances = resolve(term)

        single_conf = not (
            len(instances) == 0 and
            main_conf == 'HEAD' and
            term.startswith('C@') and
            len(term.split(' ')) == 1 and (
                len(complete_with) > 1 or (
                    len(complete_with) == 1 and
                    term != complete_with[0])
            )
        )

        if not single_conf:
            configs = (c for c in cms.all_conf(term.rsplit('C@')[1]) if c in states().active_configurations())
        else:
            configs = []

        return complete_with, main_conf, hosts, instances, single_conf, configs


class SearchJsonHandler(BaseAsyncHandler):
    json_response = True
    timeout = 30

    def get_data(self):
        term = self.get_argument('term', default='')
        info = self.get_argument('info', default='version')
        aggr = self.get_argument('aggr', default='da').lower() in ('true', 'yes', 'da', '1')
        show_instances = self.get_argument('show_instances', default=False)
        hosts, instances = resolve(term)
        return analyzer().instances_active(instances, info, show_instances, aggr)


class AvailabilityHandler(BaseAsyncHandler):
    template_name = "availability.html"
    timeout = 60

    def get_data(self):
        term = self.get_argument('term', '', True)
        _, instances = resolve(term)
        if not instances:
            return {}

        istates = states().istates()
        hosts_by_group = collections.defaultdict(list)

        for host in instances:
            if host in istates:
                for group, _ in istates[host]["groups"]:
                    hosts_by_group[group].append(host)

        response = []
        for group_name, host_list in hosts_by_group.iteritems():
            group_instances = {host: instances[host] for host in host_list}
            total_hosts, alive_hosts, total_instances, alive_instances = analyzer().availability_state(group_instances)
            response.append({
                "group_name": group_name,
                "alive_hosts_count": alive_hosts,
                "total_hosts_count": total_hosts,
                "alive_hosts_percent": float(alive_hosts) / total_hosts * 100.0,
                "alive_instances_count": alive_instances,
                "total_instances_count": total_instances,
                "alive_instances_percent": float(alive_instances) / total_instances * 100.0
            })
        response.sort(key=lambda x: x["alive_instances_percent"])
        return {
            "groups_list": response,
            "term": term
        }


class SuggestHandler(BaseAsyncHandler):
    json_response = True

    def get_data(self):
        term = self.get_argument('term', None, True)
        if not term:
            return []

        return cms.suggest_no_mainconf(term)[:20]


class PannelHandler(BaseAsyncHandler):
    template_name = 'panel.html'
    timeout = 30

    def get_data(self):
        idata = []
        data = deepcopy(PANELS[self.get_argument('prj', 'web-main', True)])

        for (loc, itype, term) in data:
            _, instances = resolve(term)
            idata.append((loc, itype, analyzer().instances_panel(instances)))

        return {'data': idata}


class FramedPannelHandler(BaseAsyncHandler):
    template_name = 'panel_framed.html'

    def get_data(self):
        return {
            'width': 1200, 'height': 1000,
            'src': 'panel?prj={}'.format(self.get_argument('prj', 'web-main', True))
        }


class FramedCPanelHandler(BaseAsyncHandler):
    template_name = 'panel_framed.html'

    def get_data(self):
        return {'width': 2000, 'height': 5000, 'src': 'cpanel'}


class ShardSizeHandler(BaseAsyncHandler):
    template_name = 'shardsize.html'
    timeout = 5

    def get_data(self):
        data = {'epoch1': int(self.get_argument('epoch1', 1396423177, True)),
                'epoch2': int(self.get_argument('epoch2', 1397032681, True)),
                'file_name': self.get_argument('file_name', 'indexinv', True)}

        histograms = states().file_sizes_histogram(data['epoch1'], data['epoch2'], data['file_name'])

        data['zones'] = {
            z: {
                'a': {'data': json_encode(histograms[z][0])},
                'b': {'data': json_encode(histograms[z][1])},
                'start': histograms[z][2],
                'step': histograms[z][4]
            } for z in histograms
            }

        return data


class DocCountHandler(BaseAsyncHandler):
    template_name = 'doccount.html'
    timeout = 10

    def get_data(self):
        file_name = self.get_argument('file_name', 'indexfrq', True)

        return {'zones': ['PlatinumTier0', 'WebTier0', 'WebTier1'],
                'ret': states().doc_count(file_name)}


class NetworkViewsHandler(BaseAsyncHandler):
    template_name = 'network_views.html'
    timeout = 10

    def get_data(self):
        version = self.get_argument('version', '3', True)
        variants = [
            {'args': {'proto': 'bb4'}, 'human_name': 'bb4'},
            {'args': {'proto': 'bb6'}, 'human_name': 'bb6'},
            {'args': {'proto': 'fb6'}, 'human_name': 'fb6'},
        ]
        default_args = {'version': version}
        panels = [
            {
                'view_name': 'framed_abnetwork',
                'human_name': 'dc to dc, bb4 and bb6',
                'args': default_args
            },
            {
                'view_name': 'framed_abnetwork_',
                'human_name': 'dc to dc',
                'args': default_args,
                'variants': variants
            },
            {
                'view_name': 'framed_pnetwork',
                'human_name': 'within dc',
                'args': default_args
            },
            {
                'view_name': 'network_graphs',
                'human_name': 'within dc, graphs',
                'args': default_args
            },
            {
                'view_name': 'framed_network',
                'human_name': 'within lines',
                'args': default_args
            },
            {
                'view_name': 'framed_network_',
                'human_name': 'broken dc',
                'args': dict(default_args, type='dc'),
                'variants': variants
            },
            {
                'view_name': 'framed_network_',
                'human_name': 'broken lines',
                'args': dict(default_args, type='line'),
                'variants': variants
            },
            {
                'view_name': 'framed_network_',
                'human_name': 'broken switches',
                'args': dict(default_args, type='switch'),
                'variants': variants
            },
        ]
        return {'views': states().network_views(),
                'panels': panels,
                'term': self.get_argument('term', '', True),
                'version': version}


class NetworkHandler(BaseAsyncHandler):
    template_name = 'network.html'
    timeout = 10

    def get_data(self):
        return {'net': states().network(
            expr=self.get_argument('expr', None, True),
            version=self.get_argument('version', None, True)
        )}


def hostport_to_href(hostport):
    return hostport.replace(':', '_')


def href_to_hostport(href):
    return href.replace('_', ':')


def jsonify(obj):
    if isinstance(obj, dict):
        res = {}
        for key, value in obj.items():
            res[str(key)] = jsonify(value)
    elif isinstance(obj, list):
        res = []
        for one in obj:
            res.append(jsonify(one))
    elif isinstance(obj, datetime.datetime):
        res = str(obj)
    elif obj is None:
        res = 0
    else:
        if isinstance(obj, unicode):
            obj = str(obj)
        res = obj
    return res


class OutliersChooseLocationHandler(BaseAsyncHandler):
    template_name = 'outliers_choose_location.html'
    timeout = 3

    def get_data(self):
        return {}


class OutliersBaseHandler(BaseAsyncHandler):
    timeout = 10

    @property
    def time(self):
        return str(states()._context['outliers']['timestamp']).partition('.')[0]

    @property
    def data(self):
        return states()._context['outliers']['data'][self.location]

    @property
    def location(self):
        uri = self.request.uri.split('/')
        return uri[-1]

    @property
    def is_error(self):
        return isinstance(self.data, str)

    def _get_error(self):
        return {'error': self.data, 'location': self.location, 'time': self.time}

    def _get_data(self):
        plot = self.data['plot'].copy()
        return {'plot': jsonify(plot), 'cnt': self.data['cnt'], 'outliers_cnt': len(self.data['outliers']),
                'time': self.time, 'location': self.location}

    def get_data(self):
        if self.location not in ['sas', 'man', 'msk']:
            assert not self.json_response
            self.redirect('/outliers')
        if self.is_error:
            return self._get_error()
        else:
            return self._get_data()


class OutliersHandler(OutliersBaseHandler):
    template_name = 'outliers.html'

    def _get_data(self):
        outliers = self.data['outliers'].copy()
        outliers = [dict(outliers[h], **{'hostport': h, 'href': hostport_to_href(h)}) for h in outliers]
        outliers.sort(key=lambda x: (x['times'], x['last_upd']), reverse=True)

        return dict(super(OutliersHandler, self)._get_data(), **{'outliers': outliers})


class OutliersFramedHandler(OutliersBaseHandler):
    template_name = 'outliers_framed.html'

    @property
    def location(self):
        uri = self.request.uri.split('/')
        return uri[-2]


class OutliersPlotJsonHandler(OutliersBaseHandler):
    json_response = True

    @property
    def location(self):
        return self.request.uri.split('/')[-1].partition('?')[0]

    def _get_data(self):
        if self.time != self.get_argument('time'):
            return super(OutliersPlotJsonHandler, self)._get_data()
        else:
            return {'time': self.time}


class OutliersHistoryHandler(BaseAsyncHandler):
    template_name = 'outliers_history.html'
    timeout = 30  # temporary

    @classmethod
    def validate_hostport(cls, name):
        return name.count(':') == 1

    def short_hostport_to_long(self, short_):
        long_ = short_[:].split(':')
        long_ = ''.join([long_[0], '.search.yandex.net', ':', long_[1]])
        return long_

    def get_pars(self, ser):
        res = {}
        for t, values in ser.items():
            for par in values:
                if isinstance(values[par], dict):
                    for qua, v in values[par].items():
                        res[par] = res[par] if par in res else {}
                        res[par][qua] = res[par][qua] if qua in res[par] else []
                        res[par][qua].append(v)
                else:
                    res[par] = res[par] if par in res else []
                    res[par].append(values[par])
        return res

    @staticmethod
    def location(host):
        if host[:3] in ['sas', 'man']:
            return host[:3]
        return 'msk'

    def get_data(self):
        hostport = href_to_hostport(self.get_argument('hostport'))
        if not self.validate_hostport(hostport):
            return dict(hostport=hostport, history={})
        longhostport = self.short_hostport_to_long(hostport)
        location = self.location(hostport)
        data = get_coll('basemetrics_' + location).find({'hostport': longhostport}).sort([('_id', -1)]).limit(10)
        outliers = get_coll('outliers_history_' + location).find({'hostport': hostport}).sort([('_id', -1)]).limit(10)

        outliers = {one['last_upd']: one for one in outliers}
        res = []
        for one in data:
            last_upd = datetime.datetime.fromtimestamp(float(max(one['series'].keys())))
            if last_upd in outliers:
                one['info'] = outliers[last_upd]
            else:
                one['info'] = {'last_upd': last_upd, 'times': 0, '_id': 0, 'par_qua': [], 'nearest': []}
            one['series'] = self.get_pars(one['series'])
            res.append(one)
            del one['_id'], one['info']['_id']

        res.sort(key=lambda x: x['info']['last_upd'], reverse=True)
        return dict(hostport=hostport, history=jsonify(res))


class WaitInfoHandler(BaseAsyncHandler):
    template_name = 'waitinfo.html'

    def get_data(self):
        dc = self.get_argument('dc', None)
        if not dc:
            return {'dc': '', 'records': [], 'info': {'total': '', 'unanswered': ''}}
        top = int(self.get_argument('top', default=30))
        quantile = int(self.get_argument('quantile', default=95))
        res, info = states()._context['waitinfo'][dc]
        res = waitinfo.filter_results(res, top_size=top, quantile_level=quantile)
        return {'dc': dc, 'records': res, 'info': info}


class ConfigPanelHandler(BaseAsyncHandler):
    template_name = 'cpanel.html'
    timeout = 30
    CONFIG = json.load(open('cpanel_params.json', 'r'))
    search_group = re.compile('(SAS|MAN|MSK|ALL|VLA)_[A-Z0-9_-]+')
    weights = {
        'PLATINUM': 1, 'TIER0': 2, 'TIER1': 3, 'DIVERSITY': 4, 'DIVERSITY_TK': 5
    }
    types = {'BALANCER', 'NOAPACHE', 'NMETA', 'MMETA', 'INTL2', 'INT', 'BASE'}

    @classmethod
    def match_itag(cls, name):
        if not cls.search_group.match(name):
            return False
        if 'BALANCER' in name:
            return name in cls.CONFIG['balancers']
        return True

    def conf_state(self, conf, hosts):
        ret = {'alive': 0, 'dead': 0}
        for h in hosts:
            try:
                if states().istates()[h]['c'][conf]['a'] == 1:
                    ret['alive'] += 1
                else:
                    ret['dead'] += 1
            except KeyError:
                ret['dead'] += 1
        return ret

    def find_alive_configuration(self, conf):
        hosts = resolve('C@' + conf)
        hosts = hosts[0]
        if len(hosts):
            state = self.conf_state(conf, hosts)
            if state['alive'] > 0:
                state['name'] = conf
                state['hosts'] = hosts
                return state
        return None

    def configurations(self):
        prefixes = {}
        active = states().active_configurations()
        for prefix in self.CONFIG['prefixes']:
            lst = []
            for one in active:
                if one.startswith(prefix):
                    conf = self.find_alive_configuration(one)
                    if conf:
                        lst.append(conf)
            if lst:
                prefixes[prefix] = sorted(lst, key=lambda x: x['alive'], reverse=True)[0]

        return [a['name'] for a in prefixes.values()] + ['BALANCER']

    def calc_i(self, conf_name):
        stats = {}
        for itag in self.itags_for_conf(conf_name):
            stats[itag] = {'h': {'alive': 0, 'dead': 0, 'alive_perc': 0}, 'i': {'alive': 0, 'dead': 0, 'alive_perc': 0}}
            hosts, instances = resolve('C@{} . I@{}'.format(conf_name, itag))
            for host in hosts:
                if host in states().alive():
                    stats[itag]['h']['alive'] += 1
                else:
                    stats[itag]['h']['dead'] += 1

                for inst in instances[host]:
                    try:
                        if states().istates()[host]['i'][inst]['a'] == 1:
                            stats[itag]['i']['alive'] += 1
                        else:
                            stats[itag]['i']['dead'] += 1
                    except KeyError:
                        stats[itag]['i']['dead'] += 1

            try:
                stats[itag]['i']['alive_perc'] = stats[itag]['i']['alive'] / \
                                              float(stats[itag]['i']['alive'] + stats[itag]['i']['dead']) * 100
                stats[itag]['h']['alive_perc'] = stats[itag]['h']['alive'] / \
                                              float(stats[itag]['h']['alive'] + stats[itag]['h']['dead']) * 100
            except ArithmeticError:
                pass
        return stats

    def itags_for_conf(self, conf):
        res = set()
        for i in cms.conf_itags(conf, cms.all_conf()[conf]):
            res.add(i['name'])
        return [a for a in res if self.match_itag(a)]

    @classmethod
    def sort_key(cls, itag):
        res = 0
        for typ, value in cls.weights.iteritems():
            res += value if typ in itag['name'] else 0
        return res

    @classmethod
    def parse_name(cls, name):
        name = name.split('_')
        res = {'geo': name[0], 'name': set()}
        for part in name[1:]:
            if part in cls.types:
                res['type'] = part
            elif part not in ('WEB', 'JUPITER', 'NEW'):
                res['name'].add(part)
        if not res['name']:
            res['name'] = 'SELF'
        else:
            res['name'] = '_'.join(res['name'])
        return res['name'], res['geo'], res['type']

    @classmethod
    def format_types(cls, iname, geo, typ):
        typ = typ.replace('NMETA', 'UPPER')
        return iname, geo, typ

    def get_data(self):
        d = collections.defaultdict
        res = {
            'prod': {'SAS': d(dict), 'MAN': d(dict), 'MSK': d(dict), 'ALL': d(dict), 'VLA': d(dict)},
            'hamster': {'SAS': d(dict), 'MAN': d(dict), 'MSK': d(dict), 'ALL': d(dict), 'VLA': d(dict)}
        }
        for conf_name in self.configurations():
            itags = self.calc_i(conf_name)
            for iname, itag in itags.items():
                iname, geo, typ = self.format_types(*self.parse_name(iname))
                itag['name'] = iname
                if 'hamster' in conf_name:
                    res['hamster'][geo][typ][iname] = itag
                else:
                    res['prod'][geo][typ][iname] = itag
        for t in res:
            for geo in res[t]:
                for typ in res[t][geo]:
                    res[t][geo][typ] = sorted(res[t][geo][typ].values(), key=self.sort_key)

        return res


class NetworkPanelHandler(BaseAsyncHandler):
    template_name = 'network_panel.html'
    timeout = 5

    def get_data(self):
        expr = self.get_argument('expr', None, True)
        json = self.get_argument('json', 'no', True)
        version = self.get_argument('version', None, True)
        try:
            net = states().network(json=json, expr=expr, version=version)
        except RuntimeError as e:
            self.application.log.warn('failed to get network state: %s', e.message)
            return None
        if json:
            self.application.log.info('netstate %s', net)
        return {'net': net}


class FramedNetworkHandler(BaseAsyncHandler):
    template_name = 'network_framed.html'

    def initialize(self, view_name):
        self._view_name = view_name

    def get_data(self):
        return {'query_args': urllib.urlencode(self.request.query_arguments, 1),
                'view_name': self._view_name,
                'show_footer': False}


class GraphsNetworkHandler(BaseAsyncHandler):
    template_name = 'network_graphs.html'

    def get_data(self):
        expr = self.get_argument('expr', None, True)
        version = self.get_argument('version', None, True)
        net = states().network(expr=expr, version=version)
        if 'yasm' not in net:
            self.set_status(404)
        else:
            return {'net': net, 'expr': expr, 'show_footer': False}


class NetworkABHandler(BaseAsyncHandler):
    template_name = 'network_ab.html'
    timeout = 10

    def get_data(self):
        network_type = self.get_argument('type', None, True)
        return {'net': states().network_ab(
            network_type=network_type,
            expr=self.get_argument('expr', None, True),
            version=self.get_argument('version', None, True)
        )}


class NetworkAB_Handler(BaseAsyncHandler):
    template_name = 'network_ab_.html'
    timeout = 10

    def get_data(self):
        type = self.get_argument('type', 'bb6', True)
        return {'net': states().network_ab_(network_type=type,
                                            dc_a=self.get_argument('a', None, True),
                                            dc_b=self.get_argument('b', None, True),
                                            line_a=self.get_argument('line_a', None, True),
                                            line_b=self.get_argument('line_b', None, True),
                                            depth=self.get_argument('depth', None, True),
                                            json=self.get_argument('json', 'no', True),
                                            expr=self.get_argument('expr', None, True),
                                            version=self.get_argument('version', None, True)),
                'type': type}


class Network_Handler(BaseAsyncHandler):
    template_name = 'network_.html'
    timeout = 10

    def get_data(self):
        net = states().network_(proto=self.get_argument('proto', 'bb6', True),
                                stat_type=self.get_argument('type', 'switch', True),
                                pess=self.get_argument('pess', 'opt', True),
                                alive=self.get_argument('alive', 'alive', True),
                                expr=self.get_argument('expr', None, True),
                                version=self.get_argument('version', None, True))
        return {'net': {k: v for k, v in net.iteritems() if v['bad']}}


class PingHandler(BaseAsyncHandler):
    template_name = 'ping.html'
    timeout = 1

    def get_data(self):
        return {}


def status_to_string(status):
    if status:
        return 'ok'
    if status is None:
        return 'rejected'
    return 'in-process'


class SuspensionHttpApi(BaseAsyncHandler):
    json_response = True
    timeout = 8

    def get_data(self):
        return {
            'result': [{
                'id': k,
                'hosts': v['hosts'],
                'status': status_to_string(v['status']),
                'message': v['message'],
            } for k, v in get_status().iteritems()]
        }

    def post_data(self):
        data = json.loads(self.request.body)
        dry_run = self.get_argument('dry_run', False, True)
        if dry_run == 'true':
            dry_run = True

        status, message = acquire(data['id'], list(data['hosts']), dry_run)
        return {'id': data['id'], 'status': status_to_string(status), 'hosts': data['hosts'], 'message': message}


class SuspensionHttpApi2(BaseAsyncHandler):
    json_response = True
    timeout = 8

    def get_data(self):
        taskid = str(self.request.uri).replace('/api/v1/tasks/', '')
        data = get_status(taskid)
        if not data:
            self.set_status(404)
        else:
            data = data.values()[0]
            return {
                'id': taskid,
                'hosts': data['hosts'],
                'status': status_to_string(data['status']),
                'message': data['message'],
            }

    def delete_data(self):
        taskid = str(self.request.uri).replace('/api/v1/tasks/', '')
        release(taskid)
        self.set_status(204)


class ResolveHandler(BaseHandler):
    def get(self, *args, **kwargs):
        self.render2(
            'resolve.html',
            terms={term: resolve(term) for term in ['I@testws', 'h@virgil', 'C@HEAD']}
        )


class ConfigurationHandler(BaseHandler):
    def get(self, **kwargs):
        self.write(str(analyzer().configuration_active_counter(self.get_argument('conf', '', True))))


class HostsApiHandler(BaseAsyncHandler):
    json_response = True

    def get_data(self):
        fqdn = self.request.uri.split('/')[-1]
        host = fqdn.replace('.search.yandex.net', '').replace('.yandex.ru', '')

        data = {}
        data.update(states().istates().get(host, {}))
        data.update({'lldp': self.lldp_info(fqdn)})

        if not data:
            self.set_status(404)

        return data

    def lldp_info(self, fqdn):

        result = {}

        import pymongo
        for record in pymongo.ReplicaSetConnection(
                HEARTBEAT_MONGODB.uri,
                socketTimeoutMS=1000,
                connectTimeoutMS=5000,
                replicaSet=HEARTBEAT_MONGODB.replicaset,
                read_preference=HEARTBEAT_MONGODB.read_preference
            )['heartbeat']['sepe_lldpinfo'].find({'host': fqdn}, limit=1):
            for key in sorted(record.keys()):
                if not key.startswith('lldp'):
                    continue
                result[key.replace(u'．', '.').replace('lldp.', '')] = record[key]

        return result


class GroupsApiHandler(BaseAsyncHandler):
    json_response = True

    def get_data(self):
        group, version = self.request.uri.split('/')[-2:]

        result = {'versions': {}}
        for version, group_state in states().instance_state.groups_state.group_versions(group).iteritems():
            total = sum(1 for _ in group_state.iter_all_instances())
            alive = sum(1 for _ in group_state.iter_alive_instances())
            if alive:
                result['versions'][version] = {'total': total, 'alive': alive}

        if result['versions']:
            current_version, _ = max(result['versions'].items(), key=self._group_sort_key)
            current_instances = list(topology.load_group2(group, current_version))
        else:
            current_version = None
            current_instances = []

        result['current'] = {
            'version': current_version,
            'instances': current_instances,
            'timestamp': int(states().instance_state.timestamp.strftime('%s')),
        }

        return result

    @staticmethod
    def _group_sort_key(x):
        version, data = x
        total, alive = data['total'], data['alive']

        return (1 if alive > 0.88 * total else 0), version


# RX-473
def _guess_line(host):
    prefix = host.split('-')[0]
    return {
        'man1': 'man1', 'man2': 'man2',
        'sas1': 'sas-1.1.1', 'sas2': 'sas-2.1.1',
        'vla1': 'vla-01', 'vla2': 'vla02',
    }.get(prefix)


def _get_enriched_hosts_for_yasm(hosts):
    for host in hosts:
        yield host

    for host in open('./extra_hosts').readlines():
        name, domain = host.strip().split('.', 1)
        line = _guess_line(host)
        if line:
            yield {'name': name, 'domain': '.' + domain, 'queue': line}


# e.g. api/v1/hardware/lines/fol3
class HardwareLineApiHandler(BaseAsyncHandler):
    vip = True

    def get(self):
        line = self.request.uri.split('/')[-1].replace('-', '')
        for host in _get_enriched_hosts_for_yasm(hosts_for_yasm()):
            queue = host.get('queue', '').replace('-', '')
            if queue == line:
                self.write(host['name'] + host['domain'] + '\n')


# api/v1/hardware/lines/
class HardwareLinesApiHandler(BaseAsyncHandler):
    vip = True

    def get(self):
        lines = set()
        for host in hosts_for_yasm():
            line = host.get('queue', None)
            if line and any(x in line for x in ['fol', 'sas', 'myt', 'man', 'iva', 'ugr', 'vla']) and line != 'man-3_b.1.08':
                lines.add(line)

        for line in sorted(lines):
            self.write(line + '\n')


class StatusHandler(BaseAsyncHandler):
    def get(self):
        self.write('<pre>topology version: {0}</pre>'.format(topology.get_latest_version()))


class StatsHandler(RequestHandler):
    @staticmethod
    def _maybe_int(val):
        try:
            return int(val)
        except (TypeError, ValueError):
            return None

    def get(self):
        invalid_response = self._maybe_int(self.get_argument("invalid_response", 0))
        metrics_limit_exceeded = self._maybe_int(self.get_argument("metrics_limit_exceeded", 0))
        if invalid_response == 1:
            stats_manager.incrementCounter("stats_handler.invalid_response")
        if metrics_limit_exceeded:
            stats_manager.incrementCounter("stats_handler.metrics_limit_exceeded")
        self.add_header('Content-Type', 'application/json; charset=utf-8')
        signals = stats_manager.getYasmSignals()
        self.write(json.dumps(signals))


# Experimental. Disable it in case of any suspicious behaviour.
class GencfgCommitsBase(BaseAsyncHandler):
    db = None
    timeout = 10

    def load_commits(self):
        self._ensure_db()
        commits = self.db.commits.find(sort=[('_id', -1)]).limit(300)
        tags = self.db.tags.find(sort=[('_id', -1)]).limit(150)

        data = {}  # commit:xx, test_passed:yy
        for record in commits:
            data[record['commit']] = {
                'test_passed': record.get('test_passed'),
                'time': record['_id'].generation_time.astimezone(pytz.timezone('Europe/Moscow')).strftime('%m-%d %H:%M'),
                'author': record.get('author', ''),
                'task_id': record.get('task_id', None),
            }

        for record in tags:
            if record['commit'] in data:
                data[record['commit']].update({'tag': record['tag']})

        data = [dict(data[key], commit=key) for key in data]
        return data

    def _ensure_db(self):
        if self.db is not None:
            return
        import pymongo

        self.db = pymongo.MongoReplicaSetClient(
            ALL_HEARTBEAT_C_MONGODB.uri,
            replicaset=ALL_HEARTBEAT_C_MONGODB.replicaset,
            localThresholdMS=30000,
            connectTimeoutMS=5000,
            read_preference=ALL_HEARTBEAT_C_MONGODB.read_preference
        )['topology_commits']


class GencfgCommits(GencfgCommitsBase):
    template_name = 'gencfg.html'

    def get_data(self, *args, **kwargs):
        data = self.load_commits()
        data.sort(key=lambda x: x['commit'], reverse=True)
        return dict(data=data)


class GencfgCommitsJson(GencfgCommitsBase):
    json_response = True

    def get_data(self, *args, **kwargs):
        data = self.load_commits()
        data.sort(key=lambda x: x['commit'], reverse=True)
        return {'commits': data}


class DeliveryDetails(BaseAsyncHandler):
    json_response = True
    timeout = 8

    def get_data(self):
        loc = str(self.request.uri).replace('/api/v1/delivery/', '').lower()
        if loc not in ('msk', 'man', 'sas'):
            self.set_status(404)
            return

        term = 'C@delivery{}_base-'.format('' if loc == 'msk' else '_' + loc)
        confs, _ = cms.suggest(term)

        if not confs:
            self.set_status(503)
            return {
                'error': 'No suitable configurations found'
            }

        conf = max(confs, key=self._to_tuple)
        _, instances = resolve(conf)
        conf = conf.replace('C@', '')

        instances_filter = frozenset(chain.from_iterable(inst.keys() for inst in instances.itervalues()))
        get_all = self.get_argument('get_all', '').lower() in ('true', 'yes', 'da', '1')
        data = {
            'details': analyzer().details(conf, instances_filter, get_all),
            'hosts': len(instances),
            'main_conf': conf,
        }

        return data

    @staticmethod
    def _to_tuple(conf):
        parts = conf.split('-')
        n1 = int(parts[1])
        n2 = int(parts[2].replace('newdb', '').replace('_i', '').replace('_f', '') or 0)
        return n1, n2
