import collections

# noinspection PyUnresolvedReferences
import api.hq as hq
import gevent


class HqCache(object):
    def __init__(self):
        self._instances_cache = {}  # conf: [(instance, set of tags)]

    def update(self):
        instances_cache = collections.defaultdict(list)

        for service in _EXPERIMENTAL_SERVICES:
            for instance in _request_hq_instances(service):
                for revision in instance.spec.revision:
                    conf = revision.id
                    instance_tuple = (instance.spec.node_name, instance.spec.allocation.port[0].port)
                    tags = set(revision.tags)
                    instances_cache[conf].append((instance_tuple, tags))

            gevent.sleep(1)

        if len(instances_cache) < len(_EXPERIMENTAL_SERVICES):
            raise RuntimeError('Too small hq result')

        self._instances_cache = instances_cache

    def list_configurations(self, prefix=None):
        prefix = prefix or ''

        result = {}
        for conf in self._instances_cache.keys():
            if conf.startswith(prefix):
                result[conf] = int(conf.split('-')[-1]) / 1000  # cms mtime format

        return result

    def list_instances(self, conf, itag=None):
        if conf not in self._instances_cache:
            return []

        result = []
        for instance, tags in self._instances_cache[conf]:
            if itag and itag not in tags:
                continue

            result.append(instance)

        return result


def _request_hq_instances(service):
    clusters = _find_clusters()

    for cluster in clusters:
        request = hq.hq_pb2.FindInstancesRequest()
        request.filter.service_id = service
        request.filter.ready_only = True
        request.field_mask.paths.append("spec.revision.id")

        rpc_client = hq.RequestsRpcClient(cluster.spec.endpoint.url + 'rpc/instances/')
        hq_client = hq.hq_stub.InstanceServiceStub(rpc_client)
        responce = hq_client.find_instances(request)
        for instance in responce.instance:
            yield instance


def _find_clusters():
    client = hq.RequestsRpcClient('http://federated.yandex-team.ru/rpc/federated/')
    fed_client = hq.federated_stub.FederatedClusterServiceStub(client)
    find_req = hq.federated_pb2.FindClustersRequest()
    return fed_client.find_clusters(find_req).value


_EXPERIMENTAL_SERVICES = {
    # 'production_clusterstate'  # FIXME
    # 'jupiter_base',
    # 'jupiter_c1_man_base',
    # 'jupiter_c1_sas_base',
    # 'man_web_base_hamster',
    # 'sas_jupiter_base',
    # 'sas_web_base_hamster',
    # 'vla_jupiter_base',
    # 'vla_web_base_hamster',
}


""" excerpt from responce.instance[112]:
meta {
  id: "man1-6261.search.yandex.net:10738@man_web_base_hamster"
  service_id: "man_web_base_hamster"
  version: "14750275909"
}
spec {
  allocation {
    port {
      name: "default"
      port: 10738
      protocol: "TCP"
    }
  }
  node_name: "man1-6261.search.yandex.net"
  revision {
    id: "man_web_base_hamster-1506049748802"
    tags: "a_topology_group-MAN_WEB_TIER0_JUPITER_BASE_HAMSTER"
    tags: "a_topology_stable-103-r10"
  }
  hostname: "bb-man1-6261-MAN_WEB_TIER0_JUP_0FA-10738.gencfg-c.yandex.net"
}
"""
