import unittest

from collections import MutableMapping
from types import InstanceType

from kernel.xmlrpc.proxy import StickyServerProxy


def _balancerUri():
    """
    IPVS Balancer Url.

    Retries will forward requests to different servers.
    Expected long timeouts (>10 seconds), shorter failures
    will be handled by IPVS automatically.
    """
    return 'cmsearch.yandex.ru'


def _rpc(endpoint, retries=[16, 32, 64], protocol='msgpack'):
    url = 'http://%s/%s/%s' % (_balancerUri(), protocol, endpoint)
    return StickyServerProxy('CMS-InstanceHost', url, retries, protocol)


class _Registry:
    class ListMethod:
        def __init__(self, args=[], nameAggr=False, noDict=False):
            self.args = args
            self.nameAggr = nameAggr
            self.noDict = noDict

        def registerAt(self, cls, name):
            def wraper(s, *args, **kwargs):
                if name == 'listSearchInstances':
                    kwargs['addFqdn'] = kwargs.get('addFqdn', True)
                for i in xrange(len(args)):
                    kwargs[self.args[i]] = args[i]
                func = getattr(s.serverProxy, name)
                if self.noDict:
                    funcRes = func(*args)
                else:
                    funcRes = func(kwargs)
                gen = (isinstance(i, dict) and RegistryObj(i) or i for i in funcRes)
                if self.nameAggr:
                    res = dict()
                    for i in gen:
                        res[i.name] = i
                    return res
                return gen
            wraper.__name__ = name
            setattr(cls, name, wraper)

    def __init__(self):
        self.serverProxy = _rpc('bs', retries=[90, 180, 270, 270])

    listConf = ListMethod(['name'], True)
    listHostConf = ListMethod(["host"], True, True)
    listAllShardTags = ListMethod(["conf", "host", "shardName"])
    listAllSearchInstanceTags = ListMethod(["conf", "host", "instanceName"])
    listSearchInstances = ListMethod(["conf", "shard", "shardTagName", "instanceTagName", "host", "port", "limit"])
    listShardOnHosts = ListMethod(["conf", "shardTagName"])
    listResources = ListMethod(["conf", "instanceName", "instanceTagName", "name"])

    def listSearchHosts(self, *args, **kwargs):
        return (i.host for i in self.listSearchInstances(*args, **kwargs))

    def __getattr__(self, name):
        return self.serverProxy.__getattr__(name)

    def nativeInterface(self):
        return self.serverProxy

    def listSearchShardsByHosts(self, *args, **kwargs):
        return ((i.host, i.shard) for i in self.listSearchInstances(*args, **kwargs))


class RegistryObj(MutableMapping):
    def key(self):
        return self.name

    def __init__(self, *args, **kwargs):
        self.__dict__.update(*args, **kwargs)
        fqdn = self.get('fqdn', None)
        if fqdn:
            self['host'] = fqdn

    def __getitem__(self, key):
        try:
            return self.__getattribute__(key)
        except AttributeError as ex:
            raise KeyError(str(ex))

    def __setitem__(self, key, val):
        self.__setattr__(key, val)

    def __delitem__(self, key):
        self.__delattr__(key)

    def __iter__(self):
        return iter(self.__dict__)

    def __len__(self):
        return len(self.__dict__)

    def __str__(self):
        return str(self.__dict__)

for name, value in _Registry.__dict__.items():
    if type(value) == InstanceType and isinstance(value, _Registry.ListMethod):
        value.registerAt(_Registry, name)

Registry = _Registry()


def createStickyProxy():
    return StickyServerProxy(
        "CMS-InstanceHost",
        "http://cmsearch.yandex.ru/msgpack/bs",
        [90, 180, 270, 270], protocol='msgpack', allow_none=True)


class RegistryTest(unittest.TestCase):
    def test1PriemkaCorrectnes(self):
        prodwsHosts = set((i.host for i in Registry.listSearchInstances("HEAD", instanceTagName="prodws-production")))
        priemkaHosts = set(
            (i.host for i in Registry.listSearchInstances("HEAD", instanceTagName="priemka-production"))
        )
        priemkaReplicaHosts = set(
            (i.host for i in Registry.listSearchInstances("HEAD", instanceTagName="priemka-production-replica"))
        )
        testwsHosts = set((i.host for i in Registry.listSearchInstances("HEAD", instanceTagName="testws-production")))
        # test priemka replica existance
        self.assertTrue(priemkaReplicaHosts)
        # test no intersection with production
        self.assertEquals(priemkaHosts & prodwsHosts, set())
        # test no intersection with testws
        self.assertEquals(priemkaHosts & testwsHosts, set())
        # test that replica belongs to all hosts
        self.assertEquals(priemkaReplicaHosts - priemkaHosts, set())

    def testHasHeadConf(self):
        confs = Registry.listConf()
        self.assertTrue('HEAD' in confs)
        self.assertFalse(confs["HEAD"].is_frozen)

    def testShardTags(self):
        shardTags = [t.name for t in Registry.listAllShardTags(conf="HEAD")]
        for tag in ("RusTier0", "EngTier0"):
            self.assertTrue(tag in shardTags)

    def testInstanceTags(self):
        instanceTags = [t.name for t in Registry.listAllSearchInstanceTags("HEAD")]
        for tag in ("testws-production", "priemka-production"):
            self.assertTrue(tag in instanceTags)

    def testListSearchInstances(self):
        self.assertTrue(Registry.listSearchInstances("HEAD", shardTagName="diversity"))
        self.assertTrue(
            Registry.listSearchInstances("HEAD", shardTagName="foreign", instanceTagName="testws-production")
        )

    def testNewStyleUpload(self):
        self.assertTrue(Registry.listSearchInstances("HEAD", instanceTagName="newstyle_upload"))

    def testListSearchHosts(self):
        self.assertTrue(list(
            Registry.listSearchHosts("HEAD", shardTagName="foreign", instanceTagName="testws-production")
        ))
