import datetime
from collections import namedtuple
from mongo import clusters, allocations, smooth


class ClusterType:
    on_demand = 'on_demand'
    dynamic = 'dynamic'
    static = 'static'
    capi = 'capi'
    samogon = 'samogon'
    ALL = {on_demand, dynamic, static, capi, samogon}


class ClusterMode:
    app = 'app'
    os = 'os'
    ALL = {app, os}


_GroupConf = namedtuple('_GroupConf', [
    'name',
    'ports',
    'active',
    'project_id',
    'agents_group',
    'intersect_with',
    'max_containers_cnt',
])


class Cluster(object):
    collection = clusters

    def __init__(self, name, type, mode, groups, ban_groups, **extras):
        self.name = name
        self.type = type
        self.mode = mode
        self.groups = self._parse_groups(groups)
        self.ban_groups = ban_groups
        self.extras = {key: value for key, value in extras.items()}

    @staticmethod
    def _parse_groups(groups_json):
        groups = {}
        for group, data in groups_json.items():
            groups[group] = _GroupConf(
                group,
                data.get('ports', None),
                data.get('active', True),
                data.get('project_id', None),
                data.get('agents_group', group),
                data.get('intersect_with', None),
                data.get('max_containers_cnt', None) and int(data['max_containers_cnt'])
            )
        return groups

    @classmethod
    def list(cls, doc=None):
        res = []
        for rec in cls.list_json(doc):
            if rec['type'] == ClusterType.on_demand:
                res.append(OnDemandCluster(**rec))
            elif rec['type'] == ClusterType.dynamic:
                res.append(DynamicCluster(**rec))
            elif rec['type'] == ClusterType.static:
                res.append(StaticCluster(**rec))
            elif rec['type'] == ClusterType.capi:
                res.append(CapiCluster(**rec))
            elif rec['type'] == ClusterType.samogon:
                res.append(SamogonCluster(**rec))
        return res

    @classmethod
    def list_json(cls, doc=None):
        return list(cls.collection().find(doc or {}, {'_id': 0}))


class OnDemandCluster(Cluster):
    pass


class _OtherCluster(Cluster):
    @property
    def mem_max(self):
        return self.extras['limits']['mem']['max']

    @property
    def mem_min(self):
        return self.extras['limits']['mem']['min']

    @property
    def cpu_max(self):
        """
        :return: n_cores * 100
        """
        return self.extras['limits']['cpu']['max']

    @property
    def cpu_min(self):
        """
        :return: n_cores * 100
        """
        return self.extras['limits']['cpu']['min']


class StaticCluster(_OtherCluster):
    pass


class DynamicCluster(_OtherCluster):
    pass


class CapiCluster(_OtherCluster):
    pass


class SamogonCluster(_OtherCluster):
    pass


class Allocation(object):
    collection = allocations

    def __init__(self, name, host, cluster, port, mem, cpu, live_till=None, ttl=None, **extras):
        assert live_till or ttl
        self.name = name
        self.host = host
        self.cluster = cluster
        self.port = int(port)
        self.mem = mem
        self.cpu = cpu
        self.live_till = live_till or datetime.datetime.now() + datetime.timedelta(seconds=ttl)
        self.extras = {key: value for key, value in extras.items()}

    def json(self):
        return dict(
            self.extras, name=self.name, host=self.host, mem=self.mem, port=self.port,
            cpu=self.cpu, live_till=self.live_till, cluster=self.cluster
        )

    @classmethod
    def list(cls, doc=None, json=False):
        res = []
        for rec in cls.collection().find(doc or {}, {'_id': 0}):
            rec = cls(**rec)
            rec = rec.json() if json else rec
            res.append(rec)
        return res

    def save(self, reset=False):
        doc = self.json()
        if not reset:
            doc = {'$set': doc}
        res = self.collection().update({'name': self.name}, doc, upsert=True)
        return res['updatedExisting']

    def remove(self):
        self.collection().remove({'name': self.name})

    def prolong(self, ttl):
        self.live_till += datetime.timedelta(seconds=ttl)

    @property
    def expired(self):
        return self.live_till < datetime.datetime.now()


class Smooth(_OtherCluster):
    collection = smooth

    def __init__(self, name, since, duration, **extras):
        super(Smooth, self).__init__(name, **extras)
        self.since = since
        self.duration = duration

    @classmethod
    def list(cls, doc=None):
        res = []
        for rec in cls.collection().find(doc or {}, {'_id': 0}):
            rec = cls(**rec)
            res.append(rec)
        return res
