from .mongo import containers
import datetime


def check_limits(limits):
    for key in ['cpu', 'mem']:
        if key not in limits:
            return False
        if 'max' not in limits[key]:
            return False
        if 'min' not in limits[key]:
            return False
    return True


class Container(object):
    _name = None
    _extras = None

    @staticmethod
    def fabric(**data):
        if data['mode'] == 'vm':
            data['mode'] = 'os'
        if data['mode'] == 'app':
            return AppContainer(**data)
        elif data['mode'] == 'os':
            return VMContainer(**data)

    @classmethod
    def list(cls, doc=None, json=False):
        res = []
        for rec in containers().find(doc or {}, {'_id': 0}, slave_okay=False):
            rec = Container.fabric(**rec)
            rec = rec.json() if json else rec
            res.append(rec)
        return res

    def save(self, w=2):
        res = containers().update({'name': self.name}, self.json(), upsert=True, w=w, wtimeout=10000)
        return res['updatedExisting']

    def delete(self):
        return containers().remove({'name': self.name})['n']

    def json(self):
        raise NotImplementedError

    @property
    def name(self):
        return self._name

    @property
    def cluster(self):
        return self._extras['cluster']

    @property
    def limits(self):
        return self._extras['limits']

    @property
    def host(self):
        return self._extras['host']

    @property
    def reported_at(self):
        return self._extras['reported_at']

    @property
    def active(self):
        return self._extras['active']

    @property
    def online(self):
        return (
            self.reported_at > datetime.datetime.now() - datetime.timedelta(minutes=2)
            and self.state != 'not_listed'
        )

    @property
    def state(self):
        return self._extras['state']


class VMContainer(Container):
    __extras__ = {
        'host', 'dns_name', 'cluster', 'mode', 'state',
        'limits', 'port', 'active', 'reported_at', 'usage', 'vlans'
    }

    def __init__(self, name, **extras):
        self._name = name
        self._extras = {key: value for key, value in extras.iteritems() if key in self.__extras__}
        if extras['limits'] and not check_limits(extras['limits']):
            raise ValueError
        assert 'dns_name' in self._extras

    @classmethod
    def list(cls, doc=None, json=False):
        res = []
        doc = doc or {}
        doc.update(mode='os')
        for rec in containers().find(doc, {'_id': 0}):
            rec = VMContainer(**rec)
            rec = rec.json() if json else rec
            res.append(rec)
        return res

    @property
    def dns_name(self):
        return self._extras['dns_name']

    def json(self):
        return dict(self._extras, name=self.name)

    @classmethod
    def find(cls, dns_name=None, name=None):
        if dns_name:
            res = cls.list({'dns_name': dns_name})
        else:
            res = cls.list({'name': name})
        if res:
            return res[0]
        return None


class AppContainer(Container):
    __extras__ = {
        'host', 'cluster', 'mode', 'state',
        'limits', 'active', 'reported_at', 'usage'
    }

    def __init__(self, name, **extras):
        self._name = name
        self._extras = {key: value for key, value in extras.iteritems() if key in self.__extras__}
        if extras['limits'] and not check_limits(extras['limits']):
            raise ValueError

    def json(self):
        return dict(self._extras, name=self.name)

    @classmethod
    def find(cls, name=None, host=None):
        if name:
            res = cls.list({'name': name})
        else:
            res = cls.list({'host': host})
        if res:
            return res[0]
        return None
