import datetime
import pymongo

from kernel.util.functional import memoized

from auto_reconnect import auto_reconnect
from mongo_params import INFRA_MONGODB


def allocate(source, counter, time):
    for year in xrange(time.year, time.year + 1):
        _allocate_year(_coll, source, counter, year)


@auto_reconnect
def _allocate_year(coll, source, counter, year):
    coll.update({
        '_id': _id_object(counter, source, year)
    }, {
        '$inc': _zero_year_object(year)
    },
        upsert=True
    )


@memoized
def _zero_year_object(year):
    res = {}
    days_in_year = (datetime.date(year+1, 1, 1) - datetime.date(year, 1, 1)).days
    for day in xrange(days_in_year):
        for hour in xrange(24):
            res['{}.{}.s'.format(day, hour)] = 0L
            res['{}.{}.n'.format(day, hour)] = 0

    return res


def start_bulk_op():
    bulk = _coll.initialize_ordered_bulk_op()
    return bulk



_coll = pymongo.MongoReplicaSetClient(
    INFRA_MONGODB.uri,
    connectTimeoutMS=500,
    replicaSet=INFRA_MONGODB.replicaset,
    w='majority',
    wtimeout=5000,
    read_preference=INFRA_MONGODB.read_preference,
)['series']['buckets2']


# TODO: consider create index
# db.buckets.create_index([
#     ('_id.source', pymongo.ASCENDING),
#     ('_id.name', pymongo.ASCENDING),
#     ('_id.date', pymongo.DESCENDING),
# ])


def store(source, name, value, time, bulk_op=None):
    coll = bulk_op or _coll
    # if not _update(coll, source, name, value, time):
    #     _allocate(coll, source, name, time)
    _update(coll, source, name, value, time)


def retrieve(source, counter, start, finish):
    for year in _years(start, finish):
        for pt in _retrieve(source, counter, year):
            if start <= pt[0] <= finish:
                yield pt


def aggregate_series(sources, counter, start, finish):
    aggr = []
    for source in sources:
        try:
            for date, value in retrieve(source, counter, start, finish):
                if value[1]:
                    aggr.append((date, value[0]/2))
        except RuntimeError:
            pass  # unknown group shouldn't kill the whole aggregation

    return sorted(aggr)


@memoized
def _retrieve(source, counter, year):
    doc = _retrieve_doc(source, counter, year)
    if not doc:
        raise RuntimeError('no data for %s %s %s', source, counter, year)

    result = []

    date = datetime.datetime(year, 1, 1)
    for day in xrange(1, 366):  # days are 1-based by accident
        day_doc = doc.get(str(day), {})
        for hour in xrange(24):
            value = day_doc.get(str(hour), None)
            if value is not None:
                result.append((date + datetime.timedelta(hours=hour), (value['s'], value['n'])))

        date += datetime.timedelta(days=1)

    return result

def _retrieve_doc(source, counter, year):
    for doc in _coll.find({'_id': _id_object(counter, source, year)}, {'_id': 0}):
        return doc


def _years(start, finish):
    return xrange(start.year, finish.year + 1)


@auto_reconnect
def _update(coll, source, name, value, time):
    index = _index(time)

    coll.find({
        '_id': _id_object(name, source, time.year)
    }).update({
        '$inc': {
            '{}.s'.format(index): value,
            '{}.n'.format(index): 1,
        }
    })


def _id_object(counter, source, year):
    return {
        'source': source,
        'counter': counter,
        'year': year
    }


def _index(time):
    return '{}.{}'.format(time.timetuple().tm_yday, time.hour)
