import hosts
import logging
import models as hardware_models
import vms

import libraries.utils as utils


def _search_hosts_heuristics_filter(market=False, nirvana=False):
    _non_search_prefixes = ['kiwi', 'kiki', 'blackthorn', 'ukrop', 'torreya', 'spider']

    result = {}
    for host in hosts.load_all(market):
        botprj = host['botprj']
        if botprj.startswith('Market'):  # there are Market hosts in search.yandex.net domain
            if not market:
                continue
        elif botprj.startswith('Search'):
            if any(host['name'].startswith(prefix) for prefix in _non_search_prefixes):
                continue
        elif 'Metrics' in botprj or ('Nirvana' in botprj and nirvana):
            pass  # What's next? Regular expressions?
        elif botprj == 'Infrastructure Products > Sistemy hranenija i obrabotki dannyh > YT':
            pass
        else:
            continue

        if '.vm.' in host['domain']:
            if not vms.acceptable_vm(host['name']):
                continue

        result[host['name'] + host['domain']] = host

    return result.values()


def get_hosts_data():
    data = {}
    for r in all_hosts():
        try:
            data[r['name']] = (
                r['dc'],
                r['queue'],
                r.get('switch') or r.get('switch_') or 'unknown-switch'
            )
        except KeyError:
            pass
    return data


def cpumodel_of_host(host):
    return models()[hosts_hardware()[utils.shortname(host)]['model']]


@utils.singleton
def models():
    logging.info('loading models')
    return hardware_models.load()


@utils.singleton
def all_hosts():
    logging.info('loading hosts')
    return hosts.load()


@utils.singleton
def really_all_hosts():
    logging.info('loading all hosts')
    return hosts.load_really_everything()


@utils.singleton
def hosts_hardware():
    logging.info('loading hostnames')
    return {one['name']: one for one in all_hosts()}


@utils.singleton
def hosts_hardware_fqdns():
    logging.info('loading host fqdns')
    return {one['name'] + one['domain']: one for one in really_all_hosts()}


@utils.singleton
def search_hosts():
    logging.info('loading search hosts')
    return _search_hosts_heuristics_filter(market=False)


@utils.singleton
def hosts_for_yasm():
    logging.info('loading yasm hosts')
    return _search_hosts_heuristics_filter(market=True, nirvana=True)


@utils.singleton
def lines_for_yasm():
    logging.info('loading yasm lines')

    lines = set()
    for host in hosts_for_yasm():
        line = host.get('queue', None)
        if line and any(x in line for x in ['fol', 'sas', 'myt', 'man', 'iva', 'ugr', 'vla']) and line != 'man-3_b.1.08':
            lines.add(line)

    return lines


@utils.singleton
def load_singletons():
    logging.info(len(models()))
    logging.info(len(all_hosts()))
    logging.info(len(really_all_hosts()))
    logging.info(len(hosts_hardware()))
    logging.info(len(hosts_hardware_fqdns()))
    logging.info(len(search_hosts()))
    logging.info(len(hosts_for_yasm()))
    logging.info(len(lines_for_yasm()))

    return 'DONE'
