from datetime import datetime, timedelta

import logging
import msgpack

from libraries.utils import timestamp
from libraries.hosts_renaming import rename
from state_obj import IStateV3
from online_state import OnlineState


class InstanceState(object):
    def __init__(self):
        self.timestamp = datetime.now() - timedelta(hours=4)
        self.ihosts = {}
        self.groups_state = OnlineState()

    def update(self, alive_hosts, db, sleep_method=None):
        new_states = {}

        for r in db['instancestatev3'].find(
                {'last_update': {'$gt': self.timestamp}},
                {'host': 1, '_id': 0, 'state': 1, 'last_update': 1}
        ):
            reported_fqdn = r['host']
            host = _shortname(reported_fqdn)
            last_update = timestamp(r['last_update'])

            if host in new_states and new_states[host]['t'] > last_update:
                continue

            try:
                host_state = msgpack.loads(r['state'])
                host_state['fqdn'] = reported_fqdn
                host_state['groups'] = set(_iter_groups(host_state))
                host_state['t'] = last_update
                host_state['c'] = _fix_configurations(host_state)

                for fqdn in _possible_fqdns(reported_fqdn):
                    self.groups_state.reset_host(fqdn)

                    for instance_key, instance_data in host_state.get('i', {}).iteritems():
                        instance_state = IStateV3(instance_data)
                        host_state['i'][instance_key] = instance_state

                        group_name = instance_state.get('tg', None)
                        group_version = instance_state.get('tv', None)
                        if not group_name or not group_version:
                            continue

                        self.groups_state.update(
                            group_name,
                            group_version,
                            fqdn,
                            _guess_fqdn_instance(fqdn, instance_key),
                            r['last_update'],
                        )

                        if sleep_method:
                            sleep_method()

                new_states[host] = host_state
                self.timestamp = max(self.timestamp, r['last_update'])

            except KeyError:
                logging.exception('Error loading report from %s', reported_fqdn)

        new_ihosts = self.ihosts.copy()
        new_ihosts.update(new_states)
        for h in new_ihosts.keys():
            if h not in alive_hosts:
                new_ihosts.pop(h)

        self.ihosts = new_ihosts


def _guess_fqdn_instance(fqdn, instance_key):
    parts = instance_key.split(':')
    if len(parts) != 2:
        return instance_key

    try:
        return fqdn, int(parts[1])
    except ValueError:
        return instance_key


def _fix_configurations(host_state):
    fixed_confs = {}
    for conf_name, conf_data in host_state.get('c', {}).iteritems():
        if '#' in conf_name:
            conf_name = conf_name.split('#')[1]
        fixed_confs[conf_name] = conf_data
    return fixed_confs


def _iter_groups(record):
    for instance_string, idata in record.get('i', {}).iteritems():
        tg = idata.get('tg', None)
        tv = idata.get('tv', None)
        if tg and tv:
            yield (tg, tv)


def _possible_fqdns(fqdn):
    yield fqdn
    renamed = rename(fqdn)
    if renamed != fqdn:
        yield renamed


def _shortname(host):
    return host.split('.')[0]
