import colorsys
from datetime import datetime
from math import cos, sin, atan
from .utils import dbscan, KDTree, quantile
from .array import Array


EPS_ = 0.8
EPS = {'PlatinumTier0': EPS_, 'WebTier0': EPS_, 'WebTier1': EPS_}
MIN_PART = {'PlatinumTier0': 0.05, 'WebTier0': 0.05, 'WebTier1': 0.05}
NAMES = ['PlatinumTier0', 'WebTier0', 'WebTier1']

LEN = 6


def configure(eps=EPS_):
    global EPS_
    EPS_ = eps


class NoLoadException(Exception):
    def __str__(self):
        return 'not enough fresh data to process'


class PlotFrame(object):
    def __init__(self, data, names, outliers):
        self.data = Array(data)
        self.names = names
        self.outliers = outliers

    def __add__(self, other):
        assert self.names == other.names
        data = self.data + other.data
        new = PlotFrame(data, self.names, list(set(self.outliers + other.outliers)))
        return new

    def __div__(self, other):
        assert isinstance(other, int)
        new = PlotFrame(self.data / other, self.names, self.outliers)
        return new

    @classmethod
    def median(cls, others):
        data = [one.data for one in others]
        a = Array(data)
        res = []
        for i in range(len(others[0].data)):
            temp = []
            for j in range(len(others[0].data[0])):
                temp.append(quantile(sorted(a[:, i, j][:]), 0.5))
            res.append(temp)
        return cls(res, others[0].names, others[0].outliers)

    @classmethod
    def avg(cls, others):
        return sum(others[1:], others[0]) / len(others)


class Colors(object):
    def __init__(self, cnt):
        self.cnt = cnt
        self.step = 1.0 / (cnt + 1)
        self.j = 1

    @staticmethod
    def get_color(h):
        rgb = [int(color * 255) for color in colorsys.hsv_to_rgb(h, 1, 1)]
        return '#{:02x}{:02x}{:02x}'.format(*rgb)

    def next_(self):
        h = self.step * self.j
        self.j = (self.j + 1) % (self.cnt + 1)
        return self.get_color(h)


class PartAnalyzer(object):
    def __init__(self, zone, threshold=0.3, eps=1.0, min_part=0.1):
        self.zone = zone
        self.threshold = threshold
        self.eps = eps
        self.min_part = min_part

        self.window = []
        self.plots = []

    def rotate_data(self, data):
        fi = -atan(sum(data[:, 1]) / sum(data[:, 0]))
        res = []
        for i in range(len(data)):
            res.append([])
            x, y = data[i][0], data[i][1]
            res[i] += [x * cos(fi) - y * sin(fi)]
            res[i] += [x * sin(fi) + y * cos(fi)]
        return res

    def get_core(self, names, values, labels):
        res_names = []
        res_values = []
        for i in range(len(names)):
            if labels[i] != -1:
                res_names.append(names[i])
                res_values.append(values[i])
        return res_names, res_values

    def _check(self, names, values):
        reserve = Array(values)
        values = reserve.copy()
        values = values.scale()
        clusters = dbscan(values, self.eps, self.min_part * len(values))
        core_names, core_values = self.get_core(names, reserve, clusters)
        neigh = KDTree(core_values)
        outliers = []
        for i in range(len(names)):
            if clusters[i] == -1:
                outliers.append({
                    'name': names[i],
                    'nearest': core_values[neigh.knn(values[i])[0]].tolist()
                })
        self.plots.append(PlotFrame(reserve / 1000, names, outliers))
        return outliers

    def check(self, names, values):
        outliers = self._check(names, values)
        self.window.append(outliers)

    def list_outliers(self):
        res = {}
        window = sum(self.window, [])
        for name in [one['name'] for one in window]:
            res[name] = {'times': 0, 'nearest': []}

        for lst in self.window:
            for one in lst:
                name = one['name']
                res[name]['times'] += 1
                res[name]['nearest'] += [one['nearest']]
            for name in set(res.keys()).difference(set([a['name'] for a in lst])):
                res[name]['nearest'] += [[None, None]]
        return res

    def outliers(self):
        res = self.list_outliers()
        res = {host: value for host, value in res.items() if value['times'] / float(len(self.window)) > self.threshold}
        return res


class Analyzer(object):
    def __init__(self, records, instances, par_qua):
        self.workers = {name: PartAnalyzer(name, eps=EPS[name], min_part=MIN_PART[name]) for name in NAMES}
        self.par_qua = par_qua
        for zone in NAMES:
            instances[zone] = set(sum([hostport.keys() for hostport in instances[zone].values()], []))
        self.instances = instances
        self.data = self._prepare(records)

    def _prepare(self, records):
        data = {}
        for zone in NAMES:
            data[zone] = list()
        for rec in records:
            host = self.to_short_name(rec['hostport'])
            rec['hostport'] = host
            for zone in self.instances:
                if host in self.instances[zone]:
                    data[zone].append(rec)
                    break
        return data

    @staticmethod
    def to_short_name(rec):
        host = rec[:rec.find('.')] + rec[rec.find(':'):]
        return host

    @classmethod
    def select_params(cls, data, par_qua):
        names = []
        values = []
        for rec in data:
            names.append(rec['hostport'])
            temp = [rec['series'][time_] for time_ in sorted(rec['series'].keys())]
            for i in range(len(temp)):
                tempi = temp[i]
                temp[i] = []
                for par, qua in par_qua:
                    temp[i] += [tempi[par][qua]] if qua else [tempi[par]]
            values.append(temp)

        return names, values

    def start(self):
        for zone, worker in self.workers.items():
            names, values = self.select_params(self.data[zone], self.par_qua)
            if len(names) < 10:
                raise NoLoadException()
            values = Array(values)
            for i in range(LEN):
                worker.check(names, values[:, i])

    def outliers(self):
        outliers = {}
        for worker in self.workers.values():
            out = worker.outliers()
            for one in self.data[worker.zone]:
                hostport = one['hostport']
                if hostport in out:
                    last_upd = float(max(one['series']))
                    outliers[hostport] = {'zone': worker.zone,
                                          'hostport': hostport,
                                          'last_upd': datetime.fromtimestamp(last_upd),
                                          'times': out[hostport]['times'],
                                          'nearest': out[hostport]['nearest'],
                                          'par_qua': self.par_qua,
                                          }
        return outliers

    def plot_outliers(self, plot, data, names, outliers):
        out = []
        ok = []

        for i in range(len(names)):
            if names[i] in outliers:
                out.append({'data': data[i], 'name': names[i], 'times': outliers[names[i]]['times']})
            else:
                ok.append({'data': data[i], 'name': names[i]})
        plot['ok'] += ok
        plot['out'] += out

    def plot(self):
        colors = Colors(len(NAMES))
        plots = {}
        for name in NAMES:
            color = colors.next_()
            plots[name] = {'color': color, 'out': list(), 'ok': list()}

        for worker in self.workers.values():
            plot = PlotFrame.median(worker.plots)
            outliers = worker.outliers()
            self.plot_outliers(plots[worker.zone], plot.data.tolist(), plot.names, outliers)

        return plots
