import math


def quantile(data, percent):
    if not data:
        return None
    k = (len(data) - 1) * percent
    f = math.floor(k)
    c = math.ceil(k)
    if f == c:
        return data[int(k)]
    d0 = data[int(f)] * (c - k)
    d1 = data[int(c)] * (k - f)
    return d0 + d1


def distance(a, b, dim):
    return sum((a[i] - b[i]) ** 2 for i in range(dim))


def region_query(points, point, eps):
    eps_squared = eps * eps

    result = []
    for i, pt in enumerate(points):
        dist_squared = distance(pt, point, 2)
        if dist_squared <= eps_squared and dist_squared > 0:
            result.append(i)

    return result


def dbscan(points, eps=0.5, min_points=500):
    visited = [False] * len(points)
    cluster = [0] * len(points)
    cluster_label = 0

    for i in xrange(len(points)):
        if visited[i]:
            continue

        visited[i] = True

        neighbors = region_query(points, points[i], eps)
        if len(neighbors) < min_points:
            cluster[i] = -1

        else:
            cluster_label += 1
            cluster[i] = cluster_label

            for neighbor in neighbors:
                if not visited[neighbor]:
                    visited[neighbor] = True
                    region = region_query(points, points[neighbor], eps)
                    if len(region) >= min_points:
                        neighbors += region

                if cluster[neighbor] <= 0:
                    cluster[neighbor] = cluster_label

    return cluster


class HyperRect(object):
    def __init__(self, points, dim):
        self.dim = dim
        high = points[0][0][:]
        low = points[0][0][:]
        for i in range(0, len(points)):
            for j in range(self.dim):
                point = points[i][0][j]
                if high[j] < point:
                    high[j] = point
                if low[j] > point:
                    low[j] = point
        self.high = high
        self.low = low

    def get_min_distance(self, query):
        total = 0
        for i in range(self.dim):
            delta = 0.0
            if self.high[i] < query[i]:
                delta = query[i] - self.high[i]
            elif self.low[i] > query[i]:
                delta = self.low[i] - query[i]
            total += delta * delta
        return total


class _Node(object):
    def __init__(self, left, right, value, k, ind, rect):
        self.left = left
        self.right = right
        self.value = value
        self.k = k
        self.ind = ind
        self.rect = rect

    def __str__(self):
        return str(self.ind) + ' ' + str(self.value)

    def __repr__(self):
        return self.__str__()

    @classmethod
    def build(cls, others, k, dim):
        k %= dim
        if not others:
            return None
        if len(others) == 1:
            return _Node(None, None, others[0][0], k, others[0][1], HyperRect(others, dim))

        others = sorted(others, key=lambda x: x[0][k])
        median = len(others) / 2
        return _Node(cls.build(others[:median], k+1, dim),
                     cls.build(others[median:], k+1, dim),
                     others[median][0],
                     k, -1, HyperRect(others, dim))


class Neighbors:
    def __init__(self, query, dim):
        self.lst = []
        self.query = query
        self.dim = dim

    def add(self, other):
        self.lst.append(other)

    def min(self):
        return min([distance(one.value, self.query, self.dim) for one in self.lst] + [1e15])

    def closest(self):
        return sorted(self.lst, key=lambda x: distance(x.value, self.query, self.dim))[0]


class KDTree(object):
    def __build(self, values):
        values = [(value, i) for i, value in enumerate(values)]
        return _Node.build(values, 0, self.dim)

    def __init__(self, values):
        assert len(values) > 0
        self.dim = len(values[0])
        self.root = self.__build(values)

    def _knn(self, query, node, neighbors, distanceSquared):
        if neighbors.min() > distanceSquared:
            if node.ind != -1:
                neighbors.add(node)
            else:
                dist_left = node.left.rect.get_min_distance(query)
                dist_right = node.right.rect.get_min_distance(query)

                if dist_left < dist_right:
                    self._knn(query, node.left, neighbors, dist_left)
                    self._knn(query, node.right, neighbors, dist_right)
                else:
                    self._knn(query, node.right, neighbors, dist_right)
                    self._knn(query, node.left, neighbors, dist_left)

    def knn(self, query):
        neigh = Neighbors(query, self.dim)
        self._knn(query, self.root, neigh, self.root.rect.get_min_distance(query))
        res = neigh.closest()
        return res.ind, res.value


# if __name__ == '__main__':
#     from sklearn.neighbors import KDTree as _KDTREE
#     from sklearn.datasets import make_blobs
#     values = make_blobs(2000, 2, 5)[0].tolist()
#     for i in range(10):
#         one = values.pop()
#         tree1 = KDTree(values)
#         tree2 = _KDTREE(values)
#         assert tree1.knn(one)[0] == tree2.query([one])[1][0][0]
#
