import libraries.yp_sd.yp_utils as yp_utils
import libraries.yp_sd.endpoint as endpoint
import libraries.yp_sd.endpoint_set as endpoint_set

from libraries.topology.groups import tag_to_version


YP_DC = 'xdc'


class ResolveError(RuntimeError):
    def __init__(self, message):
        self.message = message


def _endpoint_set_exists(yp_client, endpoint_set_name):
    for _ in yp_client.select_objects(
        'endpoint_set',
        filter='[/meta/id] = "{}"'.format(endpoint_set_name),
        selectors=['/meta/id'],
    ):
        return True
    return False


def _resolve_moved_to_exact_endpoint_set(yp_client, endpoint_set):
    if not _endpoint_set_exists(yp_client, endpoint_set):
        raise ResolveError('could not find endpoint_set {} in YP'.format(endpoint_set))
    return endpoint.existing_records_and_ids(yp_client, endpoint_set)


def _endpoint_set_names_for_gencfg_group(yp_client, group_name):
    ids = yp_client.select_objects(
        'endpoint_set',
        filter='[/labels/moved_from_gencfg_group] = "{}"'.format(group_name),
        selectors=['/meta/id'],
    )
    return {item[0] for item in ids}


def _resolve_moved_to_unknown_endpoint_sets(yp_client, group_name):
    endpoint_sets = _endpoint_set_names_for_gencfg_group(yp_client, group_name)
    if not endpoint_sets:
        raise ResolveError('could not find any endpoint_set for {} in {} YP'.format(group_name, YP_DC))
    result = set()
    for endpoint_set_ in endpoint_sets:
        result |= endpoint.existing_records_and_ids(yp_client, endpoint_set_)
    return result


def resolve_moved_to_yp(group_name, moved_to):
    if moved_to['endpoint_set']:
        assert moved_to['location']
        yp_client = yp_utils.make_client(moved_to['location'])
        return _resolve_moved_to_exact_endpoint_set(yp_client, moved_to['endpoint_set'])
    yp_client = yp_utils.make_client(YP_DC)
    return _resolve_moved_to_unknown_endpoint_sets(yp_client, group_name)


def group_alive_versions(group):
    yp_client = yp_utils.make_client(YP_DC)
    annotations = endpoint_set.load_endpoint_set_annotations(yp_client, group)
    return {
        tag_to_version(version['topology']): {
            'total': version['total_instances'],
            'alive': version['alive_instances'],
            'version': tag_to_version(version['topology']),
        }
        for version in annotations['all']
    }


def resolve_gencfg_group(group_name, mtn):
    yp_client = yp_utils.make_client(YP_DC)

    endpoint_set_name = endpoint_set.group_to_endpoint_set(group_name, 'mtn' if mtn else '')
    annotations = endpoint_set.load_endpoint_set_annotations(yp_client, endpoint_set_name)
    if not annotations:
        raise ResolveError('could not find group {} in {} YP'.format(group_name, YP_DC))
    if annotations.get('online', {}).get('topology', 'unknown') == 'unknown':
        raise ResolveError('group is offline\ndebug info: {}'.format(annotations))

    return endpoint.existing_records_and_ids(yp_client, endpoint_set_name)
