from collections import defaultdict
from httplib import HTTPConnection, HTTPException

import json

import time

import logging
from api.cms import Registry
from api.conductor import Conductor
from helpers import instancesToHostsDict


class CmsSource(object):
    def __init__(self):
        self.cms = Registry.nativeInterface()
        self.clusterstate_source = _ClusterstateSource()

    def list_confs(self, name=None):
        cms_confs = self.cms.listConf() if name is None else self.cms.listConf(name)
        cms_confs = {c['name']: {'mtime': c['mtime']} for c in cms_confs}

        return cms_confs

    def get_instances_with_meta(self, args_dict):
        if args_dict['conf'] == 'ONLINE':
            instances = {}
            resolve_result = self.clusterstate_source.resolve_online_group(args_dict['instanceTagName'])

            for instance in resolve_result['instances']:
                instances.setdefault(instance[0], set()).add(('none', shortname(instance[0]) + ':' + str(instance[1])))

            return {
                'instances': instances,
                'meta': {args_dict['instanceTagName']: resolve_result['meta']}
            }

        else:
            return {
                'instances': instancesToHostsDict(Registry.listSearchInstances(**args_dict)),
                'meta': {}
            }

    def get_instances(self, args_dict):
        if args_dict['conf'] == 'ONLINE':
            instances = {}
            resolve_result = self.clusterstate_source.resolve_online_group(args_dict['instanceTagName'])

            for instance in resolve_result['instances']:
                instances.setdefault(instance[0], set()).add(('none', shortname(instance[0]) + ':' + str(instance[1])))

            return instances

        return instancesToHostsDict(Registry.listSearchInstances(**args_dict))


def shortname(host):
    return host.replace('.search.yandex.net', '').replace('.yandex.ru', '')


class _ClusterstateSource(object):
    def __init__(self):
        self.client = _OnlineStateClient()

    def resolve_online_group(self, group_name):
        obj = self.client.resolve(group_name)
        return {
            'instances': obj.get('instances', {}),
            'meta': {
                'version': obj.get('version', ''),
            }
        }


class _OnlineStateClient(object):
    """Reconnects on any failure."""
    CLUSTERSTATE_URL = 'clusterstate.yandex-team.ru'

    def __init__(self):
        self.conn = None

    def resolve(self, group):
        if not group:
            return {}

        for attempt in xrange(3):
            try:
                if self.conn is None:
                    self.conn = HTTPConnection(self.CLUSTERSTATE_URL)
                    self.conn.connect()
                self.conn.request('GET', '/api/v1/groups/' + group + '/CURRENT')
                return json.load(self.conn.getresponse())['current']

            except (IOError, RuntimeError, HTTPException) as e:
                self.conn = None
                time.sleep(2 ** attempt)
                logging.warning('resolve attempt %s failed: [%s]', attempt, e)

        return {}


class HostSource(object):
    def _resolve(self, yr=None, conductor=None, dc=None, line=None):
        hosts = []

        if conductor:
            conductorResolver = Conductor()
            for cg in conductor:
                hosts.extend(conductorResolver.hosts(cg))

        if dc or line:
            raise RuntimeError('cannot list dc/line hosts')

        return set(hosts)

    def getHostsByGroups(self, groups):
        groupsToResolve = defaultdict(set)

        for g in groups:
            key = {
                'H': 'yr',
                'K': 'conductor',
                'd': 'dc',
                'l': 'line',
            }.get(g.prefix, None)

            if key is None:
                raise Exception("Impossible happened: HostSource got unknown prefix '{0}'".format(g.prefix))

            if key == 'yr':
                value = g.name.upper()
            else:
                value = g.name

            groupsToResolve[key].add(value)

        return self._resolve(**groupsToResolve)
