import string
from collections import namedtuple


Token = namedtuple('Token', ['lexpos', 'type', 'value'])
tokens = ('PREFIX', 'NAME', 'MINUS', 'TIMES', 'LBRACKET', 'RBRACKET')

lookup_table = {
    'conf': 'C',
    'hostgroup': 'H',
    'group': 'H',
    'host': 'h',
    'instancetag': 'I',
    'itag': 'I',
    'shardtag': 'S',
    'stag': 'S',
    'shard': 's',
    'k': 'K',
    'dc': 'd',
    'line': 'l',
}


def lex(s):
    # character groups
    end = '$'
    ops = {
        '-': 'MINUS',
        '.': 'TIMES',
        '[': 'LBRACKET',
        ']': 'RBRACKET',
        '(': 'LBRACKET',
        ')': 'RBRACKET',
    }
    id_start = string.ascii_letters + string.digits + '-=:._'
    id_body = id_start

    s += end
    i = 0
    while True:
        # skip
        if s[i] in string.whitespace:
            i += 1
            continue

        # op symbol
        if s[i] in ops:
            yield Token(value=s[i], type=ops[s[i]], lexpos=i)
            i += 1
            continue

        # id: prefix or name
        if s[i] in id_start:
            j = i + 1
            while s[j] in id_body:
                j += 1

            if s[j] == '@':
                yield Token(value=lookup(s[i:j]), type='PREFIX', lexpos=i)
                i = j + 1
            else:
                yield Token(value=s[i:j], type='NAME', lexpos=i)
                i = j

            continue

        if s[i] == end:
            yield Token(type='$END', value='$END', lexpos=i)
            break

        raise Exception('Invalid token "{}" at position {}'.format(s[i], i))


def lookup(prefix):
    return lookup_table.get(prefix.lower(), prefix)
