from resolverbase import ResolverBase
import unittest

from helpers import hostsDictsIntersection
from parser import parse
from expression import transform_literals, Literal


class MockSourceCms(object):
    def __init__(self, mock_data=None):
        self.mock_data = mock_data or {}

    def getInstancesByHosts(self, kwargsDict):
        key_map = {
            'conf': 'C',
            'shardTagName': 'S',
            'instanceTagName': 'I',
            'shard': 's',
            'host': 'h'
        }
        it = kwargsDict.iteritems()
        k, v = it.next()
        resultInstances = self.mock_data.get((key_map[k], v), {})
        for k, v in it:
            resultInstances = hostsDictsIntersection(
                resultInstances,
                self.mock_data.get((key_map[k], v), {})
            )

        return resultInstances

    def listConf(self, c):
        return {c: {'mtime': 0}}


class MockSourceHosts(object):
    def __init__(self, mock_data):
        self.mock_data = mock_data

    def getHostsByGroups(self, groups):
        hosts = set()
        for g in groups:
            hosts.update(self.mock_data.get((g.prefix, g.name.upper())))
        return hosts


class TestResolverFunctions(unittest.TestCase):
    def setUp(self):
        source_cms = MockSourceCms({
            ('h', 'ws35-001.yandex.ru'): {'ws35-001.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('h', 'ws35-002.yandex.ru'): {'ws35-002.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('h', 'ws35-003.yandex.ru'): {'ws35-003.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('h', 'ws35-004.yandex.ru'): {'ws35-004.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('h', 'ws35-006.yandex.ru'): {'ws35-006.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('h', 'ws35-035.yandex.ru'): {'ws35-035.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('h', 'ws35-040.yandex.ru'): {'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)}},
            ('S', 'FastTier'): {
                'ws35-001.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-002.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            },
            ('S', 'RusTier'): {
                'ws35-001.yandex.ru': {('primus2', None)},
                'ws35-006.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            },
            ('S', 'monitoring_type=combat'): {
                'ws35-001.yandex.ru': {('primus2', None)},
                'ws35-006.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            },
            ('C', 'HEAD'): {
                'ws35-001.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-002.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-003.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-004.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-006.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-035.yandex.ru': {('primus1', 'a'), ('primus2', None)},
                'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            }
        })
        source_hosts = MockSourceHosts({
            ('H', 'G1'): {'ws35-001.yandex.ru', 'ws35-002.yandex.ru', 'ws35-035.yandex.ru', 'ws35-040.yandex.ru'},
            ('H', 'G2'): {'ws35-003.yandex.ru', 'ws35-004.yandex.ru', 'ws35-040.yandex.ru'},
            ('K', 'G1'): {'ws35-001.yandex.ru', 'ws35-002.yandex.ru', 'ws35-035.yandex.ru', 'ws35-040.yandex.ru'},
            ('K', 'G2'): {'ws35-003.yandex.ru', 'ws35-004.yandex.ru', 'ws35-040.yandex.ru'},
            ('d', 'G1'): {'ws35-001.yandex.ru', 'ws35-002.yandex.ru', 'ws35-035.yandex.ru', 'ws35-040.yandex.ru'},
            ('l', 'G1'): {'ws35-001.yandex.ru', 'ws35-002.yandex.ru', 'ws35-035.yandex.ru', 'ws35-040.yandex.ru'},
        })

        def normalize_host_name(name):
            if '.' not in name:
                return name + '.yandex.ru'
            else:
                return name

        self.resolver = ResolverBase(source_cms, source_hosts, normalize_host_name)

    def test_hosts1(self):
        result = self.resolver.resolve_hosts('ws35-035 ws35-002.yandex.ru')
        self.assertEquals(set(result), {'ws35-035.yandex.ru', 'ws35-002.yandex.ru'})

    def test_hosts2(self):
        result = self.resolver.resolve_hosts('S@FastTier - (ws35-035 ws35-002)')
        self.assertEquals(set(result), {'ws35-001.yandex.ru', 'ws35-040.yandex.ru'})

    def test_hosts3(self):
        result = self.resolver.resolve_hosts('H@G1 - ws35-035')
        self.assertEquals(set(result), {'ws35-001.yandex.ru', 'ws35-002.yandex.ru', 'ws35-040.yandex.ru'})

    def test_hosts4(self):
        result = self.resolver.resolve_hosts('H@G1 - H@G2')
        self.assertEquals(set(result), {'ws35-001.yandex.ru', 'ws35-035.yandex.ru', 'ws35-002.yandex.ru'})

    def test_hosts5(self):
        result = self.resolver.resolve_hosts('H@G1 . H@G2')
        self.assertEquals(set(result), {'ws35-040.yandex.ru'})

    def test_hosts6(self):
        result = self.resolver.resolve_hosts('K@G1 . H@G2')
        self.assertEquals(set(result), {'ws35-040.yandex.ru'})

    def test_hosts7(self):
        result = self.resolver.resolve_hosts('H@G1 - K@G1')
        self.assertEquals(set(result), set())

    def test_hosts8(self):
        result = self.resolver.resolve_hosts('K@G1 - H@G1')
        self.assertEquals(set(result), set())

    def test_hosts6_2(self):
        result = self.resolver.resolve_hosts('d@G1 . H@G2')
        self.assertEquals(set(result), {'ws35-040.yandex.ru'})

    def test_hosts7_2(self):
        result = self.resolver.resolve_hosts('H@G1 - d@G1')
        self.assertEquals(set(result), set())

    def test_hosts8_2(self):
        result = self.resolver.resolve_hosts('d@G1 - H@G1')
        self.assertEquals(set(result), set())

    def test_hosts6_3(self):
        result = self.resolver.resolve_hosts('l@G1 . H@G2')
        self.assertEquals(set(result), {'ws35-040.yandex.ru'})

    def test_hosts7_3(self):
        result = self.resolver.resolve_hosts('H@G1 - l@G1')
        self.assertEquals(set(result), set())

    def test_hosts8_3(self):
        result = self.resolver.resolve_hosts('l@G1 - H@G1')
        self.assertEquals(set(result), set())

    def test_hosts9(self):
        self.assertEquals(set(self.resolver.resolve_hosts('H@g1')), set(self.resolver.resolve_hosts('H@G1')))

    def test_instances1(self):
        result = self.resolver.resolve_instances('ws35-035 . H@G1')
        self.assertEquals(result, {'ws35-035.yandex.ru': {('primus1', 'a'), ('primus2', None)}})

    def test_instances2(self):
        result = self.resolver.resolve_instances('S@RusTier . H@G1')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': {('primus2', None)},
            'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
        })

    def test_instances3(self):
        result = self.resolver.resolve_instances('S@RusTier . H@G1 - ws35-040 . C@HEAD')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': {('primus2', None)},
            'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
        })

    def test_instances4(self):
        result = self.resolver.resolve_instances('ws35-035 ws35-002 ws35-040')
        self.assertEquals(result, {
            'ws35-002.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            'ws35-035.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
        })

    def test_instances5(self):
        result = self.resolver.resolve_instances('H@G1')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            'ws35-002.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            'ws35-035.yandex.ru': {('primus1', 'a'), ('primus2', None)},
            'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
        })

    def test_instances6(self):
        result = self.resolver.resolve_instances('S@monitoring_type=combat . H@G1')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': {('primus2', None)},
            'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
        })

    def test_shards(self):
        result = self.resolver.resolve_shards('S@RusTier . H@G1')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': {'primus2'},
            'ws35-040.yandex.ru': {'primus1', 'primus2'},
        })

    def test_brackets(self):
        result = self.resolver.resolve_instances('(([[S@RusTier . H@G1))]]')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': {('primus2', None)},
            'ws35-040.yandex.ru': {('primus1', 'a'), ('primus2', None)},
        })

    def test_extra_source(self):
        result = self.resolver.resolve_instances('C@HEAD')
        self.assertEquals(result, {
            'ws35-001.yandex.ru': set(),
        })


class TestTransformLiterals(unittest.TestCase):
    def test_add_suffix(self):
        def add_suffix_to_host(literal):
            if literal.prefix == 'h':
                return Literal(literal.name + ".suffix", literal.prefix)
            return literal

        (dnf1, _) = transform_literals(parse("bla-bla K@cms yada-yada"), add_suffix_to_host)
        dnf2 = parse("bla-bla.suffix K@cms yada-yada.suffix")
        self.assertEquals(dnf1, dnf2)

    def test_add_suffix_fail(self):
        error_text = "error text"

        def add_suffix_to_host(literal):
            if literal.prefix == 'h':
                if literal.name == "bla-bla":
                    raise Exception(error_text)
                return Literal(literal.name + ".suffix", literal.prefix)
            return literal

        (dnf1, errors) = transform_literals(parse("bla-bla K@cms yada-yada"), add_suffix_to_host)
        dnf2 = parse("bla-bla K@cms yada-yada.suffix")
        self.assertEquals(dnf1, dnf2)
        self.assertEquals(len(errors), 1)
        self.assertEquals(errors["h@bla-bla"], error_text)


if __name__ == '__main__':
    unittest.main()
