import datetime
import pymongo
import bson
from .mongo import tags, groups, commits
from .utils import tag_to_version, commit_to_trunk, is_trunk, trunk_to_commit, version_to_tag, unzipped
from libraries.utils import memoize


def get_nearest_tagged_commit(commit):
    for tag in tags().find({'commit': {'$gte': commit}}, sort=[('commit', 1)], limit=1):
        return tag['commit']


@memoize
def get_commit_of_version(version):
    if is_trunk(version):
        return trunk_to_commit(version)
    else:
        for one in tags().find({'tag': version_to_tag(version)}, limit=1):
            return one['commit']
    return memoize.NotSave(None)


@memoize
def get_version_for_commit(commit):
    for one in tags().find({'commit': commit}, limit=1):
        return tag_to_version(one['tag'])
    return memoize.NotSave(commit_to_trunk(commit))


def load_group(group):
    group, commit = group[0], get_commit_of_version(group[1])
    return _load_group_by_commit(group, commit)


def get_latest_version():
    for record in tags().find({}, projection={'tag': 1}, sort=[('commit', pymongo.DESCENDING)], limit=1):
        tag = record['tag']
        return tag_to_version(tag)


def get_latest_commit():
    for record in groups().find({}, projection={'commit': 1}, sort=[('commit', pymongo.DESCENDING)], limit=1):
        commit = record['commit']
        return commit


def get_latest_success_commit():
    for record in commits().find({'test_passed': True}, projection={'commit': 1}, sort=[('commit', pymongo.DESCENDING)],
                                 limit=1):
        commit = record['commit']
        return commit


def get_gencfg_trunk_commits(leave_last_n=50, max_time_delta=datetime.timedelta(hours=6)):
    commits_ = []
    old_record = bson.ObjectId.from_datetime(datetime.datetime.utcnow() - max_time_delta)
    for record in commits().find(
        {
            'test_passed': True,
        },
        sort=[('commit', -1)]
    ):
        commits_.append(int(record['commit']))
        if record['_id'] < old_record and len(commits_) > leave_last_n:
            break
    return commits_


def load_latest_topology():
    commit = get_latest_commit()
    return load_topology(commit)


def load_group2(name, rev):
    if isinstance(rev, tuple):
        rev = rev[0] * 1000000 + rev[1]
    return set(load_group((name, rev)))


def load_topology(commit):
    res = {}
    ver = get_version_for_commit(commit)
    cursor = groups().find({'commit': commit})
    for record in cursor:
        instances = get_instances_params(record)
        res[(record['group'], ver)] = {(instance[0], instance[1]): False for instance in instances}

    return res


def load_tags():
    res = {}
    cursor = tags().find(
        {},
        {'tag': 1, 'commit': 1, 'diff_to': 1, 'fullstate': 1, '_id': 0}
    )
    for record in cursor:
        res[record['tag']] = record

    return res


@memoize
def _load_group_by_commit(group, commit):
    cursor = groups().find({
        'group': group,
        'commit': str(commit),
    },
        sort=[('commit', -1)],
        limit=1,
    )

    for record in cursor:
        return get_instances_params(record)
    return memoize.NotSave({})


def get_instances_params(record):
    try:
        res = unzipped(record['instances_params'])
    except KeyError:
        res = {one: {} for one in unzipped(record['instances'])}
    return res
