import collections

from .mongo import groups
from .groups import get_commit_of_version
from .utils import unzipped
from libraries.utils import memoize


def map_instance_to_group(instance, version):
    return _get_single_version(version)[instance]


@memoize
def _get_single_version(version):
    commit = get_commit_of_version(version)
    source = groups().find({'commit': commit}, {'_id': 0, 'group': 1, 'instances': 1, 'version': 1, 'master': 1})

    data = {}
    for record in source:
        group = str(record['group'])
        master = str(record.get('master', ''))
        for instance in unzipped(record['instances']):
            data[instance] = group, master

    return data


@memoize
def get_group_by_version(group, version):
    commit = get_commit_of_version(version)
    data = get_group_by_commit(group, commit)
    if not data:
        return memoize.NotSave({})
    res = dict(hosts=_map_limits(data))
    for key in data:
        if key not in {'instances_params', 'instances'}:
            res[key] = data[key]
    return res


def get_topology_by_version(version, sleep_method=None):
    topology = {}
    cursor = groups().find(
        {'commit': get_commit_of_version(version)},
        {
            '_id': 0, 'group': 1, 'instances_params': 1,
            'instances': 1, 'master': 1, 'mem': 1,
            'owners': 1, 'count': 1
        }
    )
    for data in cursor:
        group = dict(hosts=_map_limits(data))
        for key in data:
            if key not in {'instances_params', 'instances'}:
                group[key] = data[key]
        topology[group['group'], version] = group
        if sleep_method:
            sleep_method()
    return topology


def get_instances_by_version(version):
    topology = {}
    cursor = groups().find(
        {'commit': get_commit_of_version(version)},
        {
            '_id': 0, 'group': 1, 'instances_params': 1,
            'instances': 1
        }
    )
    for rec in cursor:
        topology[rec['group']] = _get_instances(rec)
    return topology


def _get_instances(rec):
    if 'instances_params' in rec:
        return unzipped(rec['instances_params']).keys()
    else:
        return unzipped(rec['instances'])


def map_hosts_groups_ports(data):
    hosts = collections.defaultdict(dict)
    for group in data:
        for host, port in data[group]:
            if group not in hosts[host]:
                hosts[host][group] = []
            hosts[host][group].append(port)
    return hosts


# 150 kb, 0.1 sec
@memoize
def get_all_topology_groups(version):
    commit = get_commit_of_version(version)
    if not commit:
        return memoize.NotSave(None)
    cursor = groups().find(
        {'commit': commit},
        {
            '_id': 0, 'group': 1
        }
    )
    groups_ = []
    for rec in cursor:
        groups_.append(rec['group'])
    return groups_


def _map_limits(record):
    data = {}
    try:
        instances_ = unzipped(record['instances_params'])
    except KeyError:
        instances_ = {one: {'cpu_guarantee': 0} for one in unzipped(record['instances'])}
    mem = record.get('mem', 0) / record['count'] if record['count'] != 0 else 0
    for instance in instances_:
        host = instance[0]
        if host not in data:
            data[host] = {}
        data[host][instance] = instances_[instance]
        data[host][instance]['mem_guarantee'] = mem
    return data


def get_group_by_commit(group, commit):
    cursor = groups().find({'group': group, 'commit': commit},
                           {'_id': 0, 'group': 1, 'instances_params': 1,
                            'instances': 1, 'master': 1, 'mem': 1,
                            'owners': 1, 'count': 1})

    for record in cursor:
        return record
    return {}
