from .mongo import search_map, tags
from libraries.json_diff import merge
from libraries.topology.utils import unzipped
from libraries.topology.searcher_lookup import load_instances
from libraries.utils import memoize


class MongoBrokenData(ValueError):
    pass


@memoize
def _get_tag_base_diff_commits(commit):
    base_commit, diff_commit = None, None
    rec = tags().find_one({'commit': str(commit)})
    tag = rec['tag']
    if 'fullstate' in rec:
        base_commit = rec['commit']
    elif 'diff_to' in rec:
        base_commit = rec['diff_to']
        diff_commit = rec['commit']
    else:
        raise MongoBrokenData()

    return tag, base_commit, diff_commit


def _get_base_diff_data(base_commit, diff_commit, direction, instances):
    base = {
        rec['hostport']: rec
        for rec in search_map().find({
            'commit': int(base_commit),
            'dir': direction,
            'hostport': {'$in': instances}
        })
    }
    diff = {}
    if diff_commit:
        diff = {
            rec['hostport']: rec
            for rec in search_map().find({
                'commit': int(diff_commit),
                'dir': direction,
                'hostport': {'$in': instances}
            })
        }
    return base, diff


@memoize
def get_search_map(commit, group, direction):
    tag, base_commit, diff_commit = _get_tag_base_diff_commits(commit)
    instances = load_instances(commit, group)
    if instances is None:
        return None
    instances = [
        '{}:{}'.format(instance['hostname'], instance['port'])
        for instance in instances
    ]
    base, diff = _get_base_diff_data(base_commit, diff_commit, direction, instances)

    mapping = {}
    for host_port in instances:
        if host_port in diff:
            if 'instances' in diff[host_port]:
                mapping[host_port] = unzipped(diff[host_port]['instances'], use_list=True)
            elif 'diff' in diff[host_port]:
                base_ = unzipped(base[host_port]['instances'], use_list=True)
                diff_ = unzipped(diff[host_port]['diff'], use_list=True)
                mapping[host_port] = merge(base_, diff_)
            else:
                raise MongoBrokenData()
        elif host_port in base:
            mapping[host_port] = unzipped(base[host_port]['instances'], use_list=True)
    return mapping


def load_subs(commit, host, port, direction):
    res = _load_subs(commit, '{}:{}'.format(host, port), direction)
    return res


@memoize
def _load_subs(commit, hostport, direction):
    base, diff = None, None
    rec = tags().find_one({'commit': str(commit)})

    if 'fullstate' in rec:
        base = rec['commit']
    elif 'diff_to' in rec:
        base = rec['diff_to']
        diff = rec['commit']
    else:
        return memoize.NotSave(None)

    rec = search_map().find_one({'commit': int(base), 'hostport': hostport, 'dir': direction})
    if rec:
        base = unzipped(rec['instances'], use_list=True)
    else:
        base = []
    if not diff:
        res = base
    else:
        rec = search_map().find_one({'commit': int(diff), 'hostport': hostport, 'dir': direction})
        if not rec:
            res = base
        else:
            if 'dead' in rec:
                return None
            if 'instances' in rec:
                res = unzipped(rec['instances'], use_list=True)
            else:
                diff = unzipped(rec['diff'], use_list=True)
                res = merge(base, diff)

    return res
