import uuid
import logging
import collections

import yp_utils


Record = collections.namedtuple('Record', [
    'fqdn',
    'ipv6',
    'port',
])
_RecordWithId = collections.namedtuple('_WrappedRecord', ['record', 'id'])


def existing_records_and_ids(yp_client, endpoint_set):
    endpoints = yp_client.select_objects(
        'endpoint',
        filter='[/meta/endpoint_set_id] = "{}"'.format(endpoint_set),
        selectors=['/spec', '/meta/id'],
    )
    records = {
        _RecordWithId(
            record=Record(
                fqdn=str(rec[0].get('fqdn')),
                ipv6=str(rec[0].get('ip6_address')),
                port=int(rec[0].get('port', -1000)),
            ),
            id=rec[1],
        )
        for rec in endpoints
    }
    return records


def _ids_to_remove_and_records_to_create(existing_records_and_ids, records):
    to_remove, to_create = [], []

    existing_records = {}
    for record_and_id in existing_records_and_ids:
        if record_and_id.record not in existing_records:
            existing_records[record_and_id.record] = record_and_id.id
        else:
            to_remove.append(record_and_id.id)

    for record in records - set(existing_records):
        to_create.append(record)
    for record in set(existing_records) - records:
        to_remove.append(existing_records[record])

    return to_remove, to_create


def _create_endpoints(yp_client, transaction_id, endpoint_set, records, set_ready):
    to_create = []
    for record in records:
        attributes = {
            'meta': {
                'id': str(uuid.uuid1()),
                'endpoint_set_id': endpoint_set,
            },
            'spec': {
                'fqdn': record.fqdn,
                'ip6_address': record.ipv6,
                'port': record.port,
            },
            'labels': {
                'gencfg': {
                    'exists': True,
                },
            },
        }
        if set_ready:
            attributes['status'] = {'ready': True}

        to_create.append(('endpoint', attributes))

    _log.debug('will create %s endpoints of %s', len(to_create), endpoint_set)
    for endpoints in yp_utils.paginate(to_create, 5000):
        yp_client.create_objects(
            endpoints,
            transaction_id=transaction_id,
        )


def _remove_endpoints(yp_client, transaction_id, endpoint_set, ids):
    _log.debug('will remove %s endpoints of %s', len(ids), endpoint_set)
    for id_ in ids:
        yp_client.remove_object(
            'endpoint',
            id_,
            transaction_id=transaction_id,
        )


def update_endpoints(yp_client, transaction_id, endpoint_set, records, set_ready=False):
    existing_records_and_ids_ = existing_records_and_ids(yp_client, endpoint_set)

    to_remove, to_create = _ids_to_remove_and_records_to_create(existing_records_and_ids_, records)

    _remove_endpoints(yp_client, transaction_id, endpoint_set, to_remove)
    _create_endpoints(yp_client, transaction_id, endpoint_set, to_create, set_ready)


_log = logging.getLogger(__name__)
