import socket
import logging
import datetime

import yp_utils


GENCFG_ABC_SERVICE_ID = '1175'
DEVELOPER_ABC_ROLE = '5'
VIRTUAL_ABC_ROLE = '14'
MANAGEMENT_ABC_ROLE = '1'


def _new_endpoint_sets(yp_client, new_names):
    endpoint_sets = yp_client.select_objects(
        'endpoint_set',
        filter='[/labels/gencfg/exists] = true',
        selectors=['/meta/id'],
    )
    existent = {item[0] for item in endpoint_sets}
    return set(new_names) - set(existent)


def group_to_endpoint_set(group_name, name_suffix=None):
    if name_suffix:
        return '{}.{}'.format(group_name, name_suffix)
    return group_name


def create_new_endpoint_sets(yp_client, names, name_suffix=''):
    to_create = []
    endpoint_set_to_gencfg_group = {
        group_to_endpoint_set(group_name, name_suffix): group_name
        for group_name in names
    }
    for new_endpoint_set in _new_endpoint_sets(yp_client, set(endpoint_set_to_gencfg_group)):
        attributes = {
            'meta': {
                'id': new_endpoint_set,
                'acl': [{
                    'action': 'allow',
                    'permissions': ['read', 'write'],
                    'subjects': [
                        'abc:service-scope:{}:{}'.format(GENCFG_ABC_SERVICE_ID, role_id)
                        for role_id in (DEVELOPER_ABC_ROLE, MANAGEMENT_ABC_ROLE, VIRTUAL_ABC_ROLE)
                    ] + ['okats', 'robot-gencfg'],
                }],
            },
            'labels': {
                'gencfg': {
                    'group': endpoint_set_to_gencfg_group[new_endpoint_set],
                    'topology': 'unknown',
                    'exists': True,
                },
                'supervisor': 'gencfg',
            },
            'annotations': {
                'gencfg': {},
            }
        }
        to_create.append(['endpoint_set', attributes])
    for items in yp_utils.paginate(to_create, 1000):
        yp_client.create_objects(items)
        _log.info('created %s endpoint_sets with [%s] suffix', len(items), name_suffix)


def list_endpoint_sets_topologies(yp_client):
    result = {}
    endpoint_sets = yp_client.select_objects(
        'endpoint_set',
        filter='[/labels/gencfg/exists] = true',
        selectors=['/meta/id', '/labels/gencfg/topology'],
    )
    for endpoint_name, topology in endpoint_sets:
        result[endpoint_name] = topology
    return result


def _get_history(yp_client, endpoint_set):
    prev_annotations, prev_topology = yp_client.select_objects(
        'endpoint_set',
        filter='[/meta/id] = "{}"'.format(endpoint_set),
        selectors=['/annotations/gencfg', '/labels/gencfg/topology'],
    )[0]
    if 'modification_time' in prev_annotations:
        history = prev_annotations.get('history', [])[:2]
        history.insert(0, {
            'modification_time': prev_annotations['modification_time'],
            'host': prev_annotations['host'],
            'topology': prev_topology,
        })
        return history
    return []


def update_endpoint_set(yp_client, transaction_id, endpoint_set, topology, annotations):
    annotations['modification_time'] = datetime.datetime.now().isoformat()
    annotations['host'] = socket.getfqdn()
    annotations['history'] = _get_history(yp_client, endpoint_set)
    yp_client.update_object(
        'endpoint_set',
        endpoint_set,
        set_updates=[
            {'path': '/labels/gencfg/topology', 'value': topology},
            {'path': '/annotations/gencfg', 'value': annotations}
        ],
        transaction_id=transaction_id,
    )


def load_endpoint_set_annotations(yp_client, endpoint_set):
    result = yp_client.select_objects(
        'endpoint_set',
        filter='[/meta/id] = "{}"'.format(endpoint_set),
        selectors=['/annotations/gencfg'],
    )
    if result:
        return result[0][0]
    return None


_log = logging.getLogger(__name__)
