import contextlib
import logging
import math
import os

import yp.client


CLUSTERS = ['sas-test', 'man-pre', 'sas', 'man', 'vla', 'xdc']


def _find_token():
    if 'YP_TOKEN' in os.environ:
        token = os.environ['YP_TOKEN']
        logging.info('Got token from env')
    else:
        token_file = os.path.expanduser('~/.yp/token')
        logging.info('Reading token from [%s]', token_file)
        token = open(token_file).read().strip()
    return token


def make_client(master):
    assert master in CLUSTERS
    master += '.yp.yandex.net:8090'
    logging.info('Connect to [%s]', master)
    config = {
        'token': _find_token(),
        'grpc_channel_options': {
            'max_send_message_length': 16 * 1024 * 1024,
            'max_receive_message_length': 16 * 1024 * 1024,
        },
    }
    return yp.client.YpClient(master, config=config)


@contextlib.contextmanager
def transaction(client_):
    transaction_id = client_.start_transaction()
    logging.debug('start transaction %s', transaction_id)
    try:
        yield transaction_id
    except Exception:
        raise
    else:
        client_.commit_transaction(transaction_id)
        logging.debug('commited transaction %s', transaction_id)


def paginate(lst, page_size):
    page_cnt = int(math.ceil(len(lst) / float(page_size)))
    for i in range(page_cnt):
        yield lst[i * page_size: (i + 1) * page_size]
