from collections import defaultdict, namedtuple, Counter
from datetime import datetime, timedelta
import json
import msgpack
import sys

from kernel.util.console import setProcTitle
from kernel.util import logging
from library.sky.hosts import braceExpansion

from libraries.resolver import resolve_hosts, resolve_instances
from utils import configure_log

import flask
import gevent
import gevent.monkey
import gevent.event
import gevent.pywsgi
gevent.monkey.patch_all()


app = flask.Flask(__name__)


class ResolverFutures(object):
    def __init__(self, cache):
        super(ResolverFutures, self).__init__()
        self.futures = {}
        self.cache = cache

    def wait(self, term):
        if term not in self.futures:
            self.futures[term] = gevent.event.AsyncResult()
            gevent.spawn(_resolve, term).link(self.futures[term])
        res = self.futures[term].wait()
        if self.cache.find(term) is None:
            self.cache.set(term, msgpack.dumps(res))
            del self.futures[term]


class Updater(gevent.Greenlet):
    warmup_terms = [
        'I@MSK_L7_BALANCER',
        'I@SAS_L7_BALANCER',
        'I@production_upper_sas_web',
        'I@production_noapache_sas_web',
        'I@production_upper_man_web',
        'I@production_noapache_man_web',
        'I@AMS_WEB_BALANCER',
        'I@production_upper_sas_imgs',
        'I@ALL_SEARCH',
    ]
    POPULAR_CNT = 30

    def __init__(self):
        super(Updater, self).__init__()
        self.most_frequently_asked_terms = Counter()
        self.cache = Cache()
        self.futures = ResolverFutures(self.cache)

    def update_frequent(self, term):
        if term not in self.warmup_terms and 'C@' in term:
            self.most_frequently_asked_terms.update((term,))

    def resolve(self, term):
        if self.cache.find(term) is None:
            self.futures.wait(term)
        return self.cache.find(term)

    def status(self):
        return self.cache.status()

    @staticmethod
    def async_resolve(term):
        import subprocess
        proc = subprocess.Popen([
            sys.executable, 'resolver_web_gevent.py', '--term', term],
            stdout=subprocess.PIPE
        )
        logging.info('resolved {}'.format(term))
        return proc.communicate()[0]

    def _warmup(self):
        n = len(self.most_frequently_asked_terms)
        n = n if n < self.POPULAR_CNT else self.POPULAR_CNT
        popular = [a[0] for a in self.most_frequently_asked_terms.most_common(n)]
        for term in self.warmup_terms + popular:
            self.cache.set(term, self.async_resolve(term))

    def _run(self):
        while True:
            try:
                self._warmup()
            except Exception as ex:
                logging.getLogger('resolver').error(ex)
                pass
            finally:
                gevent.sleep(250)


def _resolve(term):
    # noinspection PyBroadException
    try:
        start = datetime.now()
        hosts, instances = _do_resolve(term)
        logging.getLogger('resolver').info(
            'resolved [%s] in %.2f sec',
            term,
            (datetime.now() - start).microseconds / 1000000.0
        )
    except:
        logging.getLogger('resolver').exception('resolve [%s] failed', term, exc_info=sys.exc_info())
        hosts, instances = set(), {}

    def norm(x):
        return x.replace('.search.yandex.net', '').replace('.yandex.ru', '')

    norm_instances = defaultdict(dict)
    for h, v in instances.iteritems():
        for (s, i) in v:
            i = i.split('@', 1)[0]
            norm_instances[norm(h)][i] = s

    norm_hosts = map(norm, hosts)
    return norm_hosts, norm_instances


def _do_resolve(term):
    command = braceExpansion([term], True)

    if not term.startswith('H@') and not term.startswith('h@') and not term.startswith('d@') and not term.startswith('dc@'):
        instances = resolve_instances(command)
    else:
        instances = {}
    gevent.sleep()
    hosts = resolve_hosts(command)
    return hosts, instances


class Cache(object):
    """Dict with expiration. No size limit, no garbage collecting."""

    Item = namedtuple('Item', ['time', 'value'])

    def __init__(self, timeout=timedelta(minutes=5)):
        self.cache = {}
        self.timeout = timeout

    def find(self, key):
        if key not in self.cache or self.cache[key].time + self.timeout < datetime.now():
            return None
        return self.cache[key].value

    def set(self, key, value):
        self.cache[key] = Cache.Item(value=value, time=datetime.now())

    def status(self):
        return {item[0]: str(item[1].time) for item in self.cache.iteritems()}

    def __len__(self):
        return len(self.cache)


updater = Updater()


@app.route('/')
def get_main():
    term = flask.request.args.get('term', '')
    updater.update_frequent(term)
    data = updater.cache.find(term)
    if data is None:
        data = updater.resolve(term)
    return flask.Response(data)


@app.route('/status')
def get_status():
    return flask.Response(json.dumps(updater.status(), indent=4))


def main():
    args = parse_args()
    if args.term:
        term = args.term
        sys.stdout.write(msgpack.dumps(_resolve(term)))
        return

    @app.after_request
    def pre_request_logging(response):
        app.logger.info(' '.join([
            flask.request.remote_addr,
            flask.request.method,
            flask.request.url,
            str(response.status_code)
            ])
        )
        return response
    handler = configure_log(app='resolver', debug=args.debug, beta=args.beta)
    logging.getLogger('werkzeug').addHandler(handler)
    setProcTitle('resolver [{}]'.format(args.port))

    updater.start()
    server = gevent.pywsgi.WSGIServer(('::', args.port), app)
    server.serve_forever()


def parse_args():
    import argparse

    parser = argparse.ArgumentParser(description='Clusterstate resolver service')
    parser.add_argument('-p', '--port', default=20214, help='listen port', type=int)
    parser.add_argument('--warmup', dest='warmup', action='store_true', help='[default] warm up before port opening')
    parser.add_argument('--no-warmup', dest='warmup', action='store_false', help="don't warm up")
    parser.add_argument('--debug', dest='debug', default=False, action='store_true', help='enable fancy logging to stdout')
    parser.add_argument('--beta', dest='beta', default=False, action='store_true', help='run as beta (affects logging)')
    parser.add_argument('--term', dest='term', default=None, type=str)
    parser.set_defaults(warmup=True)
    return parser.parse_args()


if __name__ == '__main__':
    main()
