import logging
import json
from collections import defaultdict

from utils import singleton
from primus_groups import PRIMUS_GROUPS
from states import states
import cms
import re
import time


class Analyzer(object):
    @staticmethod
    def monitoring(hosts):
        def service_state(name):
            mhosts = states().monitoring()[name]

            success_count = 0
            total_count = 0
            defunct_hosts = []

            for h in hosts:
                if h not in mhosts:
                    continue

                total_count += 1
                if mhosts[h]:
                    success_count += 1
                else:
                    defunct_hosts.append(h)

            return {'success_count': success_count, 'total_count': total_count, 'defunct_hosts': defunct_hosts}

        ret = {}
        for service in ['cqueue', 'copier', 'cqudp']:
            ret[service] = service_state(service)
        return ret

    def instances_complex(self, conf, hosts):
        bsconfig_sync_disable_hosts = set()
        ret = {'instances_alive_count': 0, 'instances_total_count': _instances_total_count(hosts), 'revisions': {},
               'revisions_h': {}, 'revisions_i': {}, 'supermind': {}, 'supermind_h': {}, 'supermind_i': {},
               'shards': {}, 'shards_h': {}, 'shards_i': {}, 'bmd5': {}, 'bmd5_h': {}, 'bmd5_i': {}, 'mmd5': {},
               'mmd5_h': {}, 'mmd5_i': {}, 'crashes': self._crashes(hosts), 'hosts_total_count': len(hosts),
               'cgroups': {}, 'balancer_experiments': {}, 'balancer_experiments_i': {}, 'balancer_experiments_h': {}}

        dead_hosts = self._hosts_dead(hosts)
        ret['hosts_alive_count'] = ret['hosts_total_count'] - len(dead_hosts)

        inactive_hosts = set()
        dead_instances = set()
        active_hosts = set()
        active_instances = set()
        prepared_hosts = set()
        unprepared_hosts = set()
        shard_statuses_inst = defaultdict(set)
        shard_count = 0

        for h in hosts:
            for i in hosts[h].iterkeys():
                if h not in dead_hosts:
                    dead_instances.add(i)

                try:
                    idict = states().istates().get(h, None)
                    if not idict:
                        continue

                    if idict['bs'] == 0:
                        bsconfig_sync_disable_hosts.add(h)

                    if conf in idict['c']:
                        if idict['c'][conf]['a'] == 1:
                            active_hosts.add(h)
                        else:
                            inactive_hosts.add(h)

                        if idict['c'][conf]['p'] == 1:
                            prepared_hosts.add(h)
                        else:
                            unprepared_hosts.add(h)

                    if idict['i'][i]['a']:
                        inst = idict['i'][i]
                        if inst.get('t', '') == 'shardtool' and 'res' in inst:
                            res = inst['res']
                            for _, shard_status, attempt in parse_shard_statuses(res.get('shard_status', ''),
                                                                                 yield_idle=True):
                                shard_statuses_inst[shard_status].add(i)
                                shard_count += 1

                        ret['instances_alive_count'] += 1
                        dead_instances.discard(i)

                        cgroup = states().istates()[h]['i'][i].get('c', 'unknown')
                        if cgroup is None:
                            cgroup = 'none'
                        cgroup = cgroup.replace('\n', '-')

                        port = i.split(':')[1]
                        cgroup = cgroup.replace('slot' + port, 'slotPORT')

                        ret['cgroups'].setdefault(cgroup, set())
                        ret['cgroups'][cgroup].add(i)

                        key = str(idict['i'][i].get('u', ''))
                        key = key.replace('svn+ssh://zomb-sandbox-ro@arcadia.yandex.ru/arc/tags/', '')
                        key = key.replace('svn+ssh://zomb-sandbox-ro@arcadia.yandex.ru/arc/branches/', '')
                        key = key.replace('svn+ssh://arcadia.yandex.ru/arc/tags/', '')
                        key = key.replace('svn+ssh://arcadia.yandex.ru/', '')
                        ret['revisions'].setdefault(key, 0)
                        ret['revisions'][key] += 1
                        ret['revisions_i'].setdefault(key, set())
                        ret['revisions_i'][key].add(i)
                        ret['revisions_h'].setdefault(key, set())
                        ret['revisions_h'][key].add(h)

                        uname = ''
                        oss = states().oss()
                        if h in oss:
                            uname = oss[h]['os']
                        key = str(idict['i'][i].get('md5', '')) + '#' + str(uname)
                        ret['bmd5'].setdefault(key, 0)
                        ret['bmd5'][key] += 1
                        ret['bmd5_i'].setdefault(key, set())
                        ret['bmd5_i'][key].add(i)
                        ret['bmd5_h'].setdefault(key, set())
                        ret['bmd5_h'][key].add(h)

                        shard = idict['i'][i]['s'].split('-')[-1]
                        if shard == '.':  # hack for realsearches
                            shard = 'none'
                        cms_shard = hosts[h][i].split('-')[-1]
                        key = shard + '#' + cms_shard
                        ret['shards'].setdefault(key, 0)
                        ret['shards'][key] += 1
                        ret['shards_i'].setdefault(key, set())
                        ret['shards_i'][key].add(i)
                        ret['shards_h'].setdefault(key, set())
                        ret['shards_h'][key].add(h)

                        key = str(idict['i'][i].get('mv', 'unknown'))
                        ret['mmd5'].setdefault(key, 0)
                        ret['mmd5'][key] += 1
                        ret['mmd5_i'].setdefault(key, set())
                        ret['mmd5_i'][key].add(i)
                        ret['mmd5_h'].setdefault(key, set())
                        ret['mmd5_h'][key].add(h)

                        if idict['i'][i]['mu'] != 0.0 and idict['i'][i]['mu'] != 1.0:
                            key = str(idict['i'][i]['mu'])
                        else:
                            key = '1.0'
                        ret['supermind'].setdefault(key, 0)
                        ret['supermind'][key] += 1
                        ret['supermind_i'].setdefault(key, set())
                        ret['supermind_i'][key].add(i)
                        ret['supermind_h'].setdefault(key, set())
                        ret['supermind_h'][key].add(h)

                        if inst.get('res', None) and inst['res'].get('balancer_exp', None):
                            for key in inst['res']['balancer_exp'].iterkeys():
                                ret['balancer_experiments'].setdefault(key, 0)
                                ret['balancer_experiments'][key] += 1
                                ret['balancer_experiments_i'].setdefault(key, set())
                                ret['balancer_experiments_i'][key].add(i)
                                ret['balancer_experiments_h'].setdefault(key, set())
                                ret['balancer_experiments_h'][key].add(h)

                except KeyError as e:
                    logging.exception(e)

        ret['r_keys'] = sorted(ret['revisions'], key=lambda x: ret['revisions'][x], reverse=True)
        ret['s_keys'] = sorted(ret['shards'], key=lambda x: ret['shards'][x], reverse=True)
        ret['b_keys'] = sorted(ret['bmd5'], key=lambda x: ret['bmd5'][x], reverse=True)
        ret['m_keys'] = sorted(ret['mmd5'], key=lambda x: ret['mmd5'][x], reverse=True)
        ret['u_keys'] = sorted(ret['supermind'], key=lambda x: ret['supermind'][x], reverse=True)
        ret['be_keys'] = sorted(ret['balancer_experiments'], key=lambda x: ret['balancer_experiments'][x], reverse=True)
        ret['bsc_disable'] = len(list(bsconfig_sync_disable_hosts))
        ret['bsc_disable_hosts'] = sorted(bsconfig_sync_disable_hosts)
        ret['c_lens'] = {x: len(ret['cgroups'][x]) for x in ret['cgroups'].keys()}
        ret['c_keys'] = sorted(ret['c_lens'], key=lambda x: ret['c_lens'][x], reverse=True)

        for k in ret['m_keys']:
            ret['mmd5_h'][k] = sorted(ret['mmd5_h'][k])
            ret['mmd5_i'][k] = sorted(ret['mmd5_i'][k])
        for k in ret['b_keys']:
            ret['bmd5_h'][k] = sorted(ret['bmd5_h'][k])
            ret['bmd5_i'][k] = sorted(ret['bmd5_i'][k])
        for k in ret['s_keys']:
            ret['shards_h'][k] = sorted(ret['shards_h'][k])
            ret['shards_i'][k] = sorted(ret['shards_i'][k])
        for k in ret['r_keys']:
            ret['revisions_h'][k] = sorted(ret['revisions_h'][k])
            ret['revisions_i'][k] = sorted(ret['revisions_i'][k])
        for k in ret['u_keys']:
            ret['supermind_h'][k] = sorted(ret['supermind_h'][k])
            ret['supermind_i'][k] = sorted(ret['supermind_i'][k])
        for k in ret['be_keys']:
            ret['balancer_experiments_h'][k] = sorted(ret['balancer_experiments_h'][k])
            ret['balancer_experiments_i'][k] = sorted(ret['balancer_experiments_i'][k])
        for k in ret['c_keys']:
            ret['cgroups'][k] = sorted(ret['cgroups'][k])

        ret['active_hosts'] = sorted(active_hosts)
        ret['ahc'] = len(active_hosts)
        ret['inactive_hosts'] = sorted(inactive_hosts)
        ret['ihc'] = len(inactive_hosts)
        ret['prepared_hosts'] = sorted(prepared_hosts)
        ret['phc'] = len(prepared_hosts)
        ret['unprepared_hosts'] = sorted(unprepared_hosts)
        ret['uhc'] = len(unprepared_hosts)
        ret['dead_hosts'] = sorted(dead_hosts)
        ret['dhc'] = len(dead_hosts)

        ret['dead_instances'] = sorted(dead_instances)
        ret['dic'] = len(dead_instances)
        ret['active_instances'] = sorted(active_instances)
        ret['aic'] = len(active_instances)
        ret['thc'] = len(set(hosts.keys()))
        ret['tic'] = _instances_total_count(hosts)

        if shard_statuses_inst:
            ret['shardtool'] = build_shardtool_view(shard_statuses_inst)
            ret['shardtool']['shard_count'] = shard_count

        return ret

    def configuration_simple(self, conf, hosts):
        ret = {'hosts_inactive_count': 0, 'hosts_active_count': 0, 'hosts_active': set(), 'shards': set()}

        for h in hosts:
            try:
                if states().istates()[h]['c'][conf]['a'] == 1:
                    ret['hosts_active_count'] += 1
                    ret['hosts_active'].add(h)
                else:
                    ret['hosts_inactive_count'] += 1
            except KeyError:
                ret['hosts_inactive_count'] += 1
            for i in hosts[h].iterkeys():
                ret['shards'].add(hosts[h][i].split('-')[-1])
        ret['hosts_active'] = sorted(ret['hosts_active'])[:100]
        ret['shards'] = list(ret['shards'])

        ret.update(self.instances_simple(hosts))

        return ret

    @staticmethod
    def configuration_active_counter(conf):
        ret = 0

        for h in states().istates():
            try:
                if states().istates()[h]['c'][conf]['a'] == 1:
                    ret += 1
            except KeyError:
                pass

        return ret

    @staticmethod
    def _hosts_dead(hosts):
        ret = set()
        for h in hosts:
            if h not in states().alive():
                ret.add(h)
        return ret

    @staticmethod
    def instances_simple(hosts):
        ret = {'instances_alive_count': 0, 'instances_total_count': _instances_total_count(hosts)}

        for h in hosts:
            for i in hosts[h].iterkeys():
                try:
                    if states().istates()[h]['i'][i]['a'] == 1:
                        ret['instances_alive_count'] += 1

                except KeyError:
                    pass
        return ret

    def configuration(self, conf, hosts):
        ret = {'hosts_dead': self._hosts_dead(hosts), 'hosts_inactive': set(), 'hosts_active': set()}
        for h in hosts:
            try:
                if states().istates()[h]['c'][conf]['a'] == 1:
                    ret['hosts_active'].add(h)
                else:
                    ret['hosts_inactive'].add(h)
            except KeyError:
                ret['hosts_inactive'].add(h)
        return ret

    @staticmethod
    def instances(hosts):
        ret = {'instances_alive': set(), 'instances_dead': set()}
        for h in hosts:
            for i in hosts[h].iterkeys():
                try:
                    if states().istates()[h]['i'][i]['a'] == 1:
                        ret['instances_alive'].add(i)
                    else:
                        ret['instances_dead'].add(i)
                except KeyError:
                    ret['instances_dead'].add(i)
        return ret

    @staticmethod
    def bsconfig(hosts):
        ret = {'bsc_sync_enable_hosts': set(), 'bsc_sync_disable_hosts': set()}
        for h in hosts:
            try:
                if states().istates()[h]['bs'] == 1:
                    ret['bsc_sync_enable_hosts'].add(h)
                else:
                    ret['bsc_sync_disable_hosts'].add(h)
            except KeyError:
                ret['bsc_sync_disable_hosts'].add(h)
        return ret

    @staticmethod
    def _crashes(hosts):
        ret = [0., 0., 0.]
        for h in hosts:
            for i in hosts[h].iterkeys():
                for e in [0, 1, 2]:
                    try:
                        ret[e] += states().crashes()[e][i]
                    except KeyError:
                        pass
        k = 1. / _instances_total_count(hosts) if _instances_total_count(hosts) > 0 else 0
        for e, i in [(0, 2 - 1), (1, 6 - 1), (2, 16 - 1)]:
            ret[e] = ret[e] * k / i
        return ret

    def instances_panel(self, hosts):
        ret = self.instances_simple(hosts)
        ret['crashes'] = self._crashes(hosts)
        ret['hosts_total_count'] = len(hosts)
        ret['hosts_alive_count'] = ret['hosts_total_count'] - len(self._hosts_dead(hosts))
        return ret

    @staticmethod
    def availability_state(instances_dict):
        # same as other 9001 functions in this class but a little faster for what it does
        total_hosts, alive_hosts, total_instances, alive_instances = 0, 0, 0, 0

        alive_state = states().alive()
        istates = states().istates()

        for host, host_instances in instances_dict.iteritems():
            total_hosts += 1
            total_instances += len(host_instances)
            if host in alive_state:
                alive_hosts += 1

            if host not in istates:
                continue

            instances_data = istates[host]['i']
            for instance in host_instances:
                if instance in instances_data and instances_data[instance]['a'] == 1:
                    alive_instances += 1

        return total_hosts, alive_hosts, total_instances, alive_instances

    def skyinfo(self, hosts):
        services = ['cqueue', 'copier', 'cqudp']
        ret = {x: {'hosts_alive_count': 0, 'hosts_defunct': set()} for x in services}
        ret['hosts_dead'] = self._hosts_dead(hosts)

        for h in hosts:
            if h in ret['hosts_dead']:
                continue

            for service in services:
                try:
                    if states().skyinfo()[h][service]:
                        ret[service]['hosts_alive_count'] += 1
                    else:
                        ret[service]['hosts_defunct'].add(h)
                except KeyError:
                    ret[service]['hosts_defunct'].add(h)

        return ret

    def versions(self, hosts):
        hosts_dead = self._hosts_dead(hosts)

        versions = defaultdict(set)

        for h in hosts:
            if h.startswith('fb-sas1-') or h.startswith('fb-man1-'):
                continue

            if h in hosts_dead or h not in states().oss():
                key = 'Dead', 'none'
            else:
                key = states().oss()[h]['os'], states().oss()[h]['ver']
            versions[key].add(h)

        for v in versions:
            versions[v] = -len(versions[v]), versions[v]

        return versions

    def versions2(self, hosts):
        hosts_dead = self._hosts_dead(hosts)
        versions = {'old_linux': set(), 'web_ubuntu': set(), 'unknown': set(), 'dead': set(), 'old_freebsd': set()}

        for h in hosts:
            if h.startswith('fb-sas1-') or h.startswith('fb-man1-'):
                continue

            if h in hosts_dead or h not in states().oss():
                versions['dead'].add(h)
                continue

            os = states().oss()[h]['os'].lower()

            if os == 'linux':
                if h not in states().softinfo():
                    versions['unknown'].add(h)
                elif states().softinfo()[h]['tejblum'] > 0:
                    versions['old_linux'].add(h)
                else:
                    versions['web_ubuntu'].add(h)
            elif os == 'freebsd':
                versions['old_freebsd'].add(h)
            else:
                versions.setdefault(os, set())
                versions[os].add(h)

        for v in versions:
            versions[v] = -len(versions[v]), versions[v]

        return versions

    def versions3(self, hosts):
        hosts_dead = self._hosts_dead(hosts)
        versions = dict(dead=set())

        for h in hosts:
            if h.startswith('fb-sas1-') or h.startswith('fb-man1-'):
                continue

            if h in hosts_dead:
                versions['dead'].add(h)
                continue

            golovan = states().golovan().get(h, 'unknown')
            versions.setdefault(golovan, set())
            versions[golovan].add(h)

        for v in versions:
            versions[v] = -len(versions[v]), versions[v]

        return versions

    def versions4(self, hosts):
        hosts_dead = self._hosts_dead(hosts)
        versions = dict(dead=set())

        for h in hosts:
            if h.startswith('fb-sas1-') or h.startswith('fb-man1-'):
                continue

            if h in hosts_dead:
                versions['dead'].add(h)
                continue

            iss = states().iss().get(h, 'unknown')
            versions.setdefault(iss, set())
            versions[iss].add(h)

        for v in versions:
            versions[v] = -len(versions[v]), versions[v]

        return versions

    def packages(self, hosts):
        packages = {}

        for host, dpkg in states().packages().iteritems():
            if host in hosts:
                for name, ver in dpkg.items():
                    if name.startswith('yandex-'):
                        packages.setdefault(name, {}).setdefault(ver, set()).add(host)

        dead_hosts = self._hosts_dead(hosts)
        for p in packages.values():
            none = set(hosts)
            for v in p.values():
                none -= v
            p['none'] = none - dead_hosts
            p['dead'] = dead_hosts

            for v in p:
                p[v] = -len(p[v]), p[v]

        return packages

    def _deploy(self, instances):
        # instances = {'man1-2000:12345': {
        #     'tags': ['a_tier_WebTier0', 'a_itype_base'],
        #     'shards': {'sh1': 1454683735, 'sh2': 1454683735}}
        # }

        ts = 0
        ds = 0
        reported_shards = 0

        shards = defaultdict(lambda: {'total': 0, 'not_installed': 0, 'zone': 'other'})
        spams = defaultdict(int)

        for name, instance in instances.iteritems():
            host = name.split(':')[0]
            tier = get_tier_by_tags(instance['tags'])

            for s, shard_ts in instance['shards'].iteritems():
                shards[s]['zone'] = tier or s
                shards[s]['total'] += 1

                # no report about host or shard on host
                if host not in states().states() or s not in states().states()[host]:
                    shards[s]['not_installed'] += 1
                    continue

                state = states().states()[host][s]
                reported_shards += 1

                # timestamp in CMS not matched with reported one (shard was reinitialized)
                if state['ct'] < shard_ts:
                    shards[s]['not_installed'] += 1
                    continue

                # spam version
                if state['sv'] > 0:
                    spams[get_spam_time(state['sv'])] += 1

                if not state['i']:
                    shards[s]['not_installed'] += 1

                ts += state['ts']
                ds += state['ds']

        total = sum(x['total'] for x in shards.itervalues())
        installed = total - sum(x['not_installed'] for x in shards.itervalues())
        normalized_ts = (float(ts) / reported_shards * total) if reported_shards > 0 else 0
        if ds == 0 and normalized_ts == 0:
            normalized_ts = 1
        first_replica = sum(1 for x in shards.itervalues() if x['total'] > x['not_installed'])
        second_replica = sum(1 for x in shards.itervalues() if
                             x['total'] - x['not_installed'] >= 2 or x['total'] == 1 and x['not_installed'] == 0)
        all_replicas = sum(1 for x in shards.itervalues() if x['not_installed'] == 0)

        groups = defaultdict(lambda: {'total': 0, 'installed': 0})
        zones = defaultdict(lambda: {'total': 0, 'installed': 0})
        primuses = defaultdict(lambda: {'total': 0, 'installed': 0})

        for s, d in shards.iteritems():
            g = get_primus_group(s)
            groups[g]['total'] += 1

            zones[d['zone']]['total'] += 1

            p = s.split('-', 1)[0]
            primuses[p]['total'] += 1

            if d['total'] > d['not_installed']:
                groups[g]['installed'] += 1
                zones[d['zone']]['installed'] += 1
                primuses[p]['installed'] += 1

        for s in spams:
            spams[s] = -spams[s]

        return {
            'spams': spams,
            'primuses': primuses,
            'tiers': zones,
            'groups': groups,
            'group_keys': sorted(groups.keys()),
            'total': total,
            'installed': installed,
            'normalized_ts': normalized_ts,
            'ds': ds,
            'first_replica': first_replica,
            'second_replica': second_replica,
            'all_replicas': all_replicas,
            'unique': len(shards),
        }

    # Doesn't work
    def deploy(self, conf, instances):
        return self._deploy(cms.get_full_instances(conf, instances))

    def _details(self, instances, get_all):
        # instances = {'man1-2000:12345': {
        #     'tags': ['a_tier_WebTier0', 'a_itype_base'],
        #     'shards': {'sh1': 1454683735, 'sh2': 1454683735}}
        # }

        shards = defaultdict(lambda: {
            'hosts': defaultdict(lambda: {
                'instances': defaultdict(lambda: False),
                'ds': 0,
                'ts': 0,
                'a': False,
                'i': False,
                'spam': '',
            }),
        })

        for name, instance in instances.iteritems():
            host = name.split(':')[0]
            tier = get_tier_by_tags(instance['tags'])

            for shard, shard_ts in instance['shards'].iteritems():
                shards[shard]['hosts'].setdefault(host, {
                    'instances': defaultdict(lambda: False),
                    'ds': 0,
                    'ts': 0,
                    'a': False,
                    'i': False,
                    'spam': '',
                })
                shards[shard]['zone'] = tier or shard

                try:
                    shards[shard]['hosts'][host]['instances'][name] = states().istates()[host]['i'][name]['a']
                except KeyError:
                    shards[shard]['hosts'][host]['instances'][name] = False

                try:
                    state = states().states()[host][shard]
                except KeyError:
                    continue

                if state['ct'] < shard_ts:
                    continue

                if state['sv'] > 0:
                    shards[shard]['hosts'][host]['spam'] = get_spam_time(state['sv'])

                shards[shard]['hosts'][host]['i'] = state['i']
                shards[shard]['hosts'][host]['a'] = state['a']
                shards[shard]['hosts'][host]['ts'] = state['ts'] / 1024. ** 3
                shards[shard]['hosts'][host]['ds'] = state['ds'] / 1024. ** 3
                if state['a']:  # bug in agent
                    shards[shard]['hosts'][host]['ds'] = shards[shard]['hosts'][host]['ts']

        for s in shards:
            shards[s]['group'] = 'gr #{0}'.format(get_primus_group(s))
            shards[s]['icount'] = sum(len(shards[s]['hosts'][h].get('instances', {})) for h in shards[s]['hosts'])

            installed = 0
            total = 0
            for host_name, host in shards[s]['hosts'].iteritems():
                total += 1
                if host['i']:
                    installed += 1
                host['dead'] = host_name not in states().alive()

            shards[s]['order'] = (installed + 1e-6) / total

        return list(prepare(shards, get_all))

    # Doesn't work
    def details(self, conf, instances_filter, get_all):
        return self._details(cms.get_full_instances(conf, instances_filter), get_all)

    def _shardtool(self, instances):
        shards = {}
        for instance_name in instances:
            host = instance_name.split(':')[0]

            try:
                report = states().istates()[host]['i'][instance_name]

                if report.get('t', '') == 'shardtool' and 'res' in report:
                    res = report['res']
                    for shard_name, status, attempt in parse_shard_statuses(res.get('shard_status', '')):
                        shards.setdefault(shard_name, {'hosts': {}})
                        shards[shard_name]['hosts'].setdefault(host, {})
                        shards[shard_name]['hosts'][host].setdefault('instances', {})
                        shards[shard_name]['hosts'][host]['instances'][instance_name] = {
                            'status': status,
                            'attempt': attempt,
                            'status_text': get_status_text(status, attempt),
                        }

            except KeyError:
                continue

        for s in shards:
            shards[s]['icount'] = sum(len(shards[s]['hosts'][h].get('instances', {})) for h in shards[s]['hosts'])

            total = 0
            for host_name, host in shards[s]['hosts'].iteritems():
                total += 1
                host['dead'] = host_name not in states().alive()

            order = []
            for h in shards[s]['hosts']:
                for instance_name, instance in shards[s]['hosts'][h]['instances'].iteritems():
                    instance_order = get_instance_order(instance), instance_name
                    instance['order'] = instance_order
                    order.append(instance_order)

            shards[s]['order'] = sorted(order)

        return list(prepare(shards, True))

    def shardtool(self, instances):
        return self._shardtool(instances)

    @staticmethod
    def configurations_by_hosts(hosts):
        ret = {'active': {}, 'prepared': {}}

        for h in hosts:
            try:
                confs = states().istates()[h]['c']

                for c in confs.iterkeys():
                    if c.startswith('urldat_'):
                        continue

                    if confs[c]['a'] == 1:
                        # noinspection PyTypeChecker
                        ret['active'].setdefault(c, set())
                        ret['active'][c].add(h)

                    if confs[c]['p'] == 1:
                        # noinspection PyTypeChecker
                        ret['prepared'].setdefault(c, set())
                        ret['prepared'][c].add(h)

            except KeyError:
                continue

        ret['a_lens'] = {x: len(ret['active'][x]) for x in ret['active'].keys()}
        ret['a_keys'] = sorted(ret['a_lens'], key=lambda x: ret['a_lens'][x], reverse=True)
        ret['p_lens'] = {x: len(ret['prepared'][x]) for x in ret['prepared'].keys()}
        ret['p_keys'] = sorted(ret['p_lens'], key=lambda x: ret['p_lens'][x], reverse=True)

        ret['topology'] = {}
        for k in ret['a_keys']:
            ret['active'][k] = sorted(ret['active'][k])
            ret['topology'][k] = ''
        for k in ret['p_keys']:
            ret['prepared'][k] = sorted(ret['prepared'][k])
            if k not in ret['topology']:
                ret['topology'][k] = ''

        return ret

    @staticmethod
    def _get_all_active_instance_data(instances):
        all_states = states().istates()
        for h, instances_by_host in instances.iteritems():
            if h in all_states:
                for i in instances_by_host.iterkeys():
                    if i in all_states[h]['i'] and all_states[h]['i'][i].get('a', 0) == 1:
                        yield (i, all_states[h]['i'][i])

    def _count_instance(self, ret, resource, data, key, instance, show_instances):
        info = self._get_version_from_url(data.get(key, None))
        if resource not in ret:
            ret[resource] = {}
        if show_instances:
            if info not in ret[resource]:
                ret[resource][info] = []
            ret[resource][info].append(instance)
        else:
            ret[resource][info] = ret[resource].get(info, 0) + 1

    @staticmethod
    def _get_version_from_url(url):
        if url is None:
            return None
        m = re.search(r'/arc/(.*)/arcadia', url)
        if m:
            return m.group(1)
        return url

    def instances_active(self, instances, info, show_instances, aggregate):
        key = 'st' if (info == 'taskid') else 'u'
        ret = {}
        for instance, instance_data in self._get_all_active_instance_data(instances):
            resources = instance_data.get('res', {})
            if not resources:
                continue

            if aggregate:
                for name, res_data in resources.iteritems():
                    self._count_instance(ret, name, res_data, key, instance, show_instances)
                self._count_instance(ret, 'binary', instance_data, key, instance, show_instances)

            else:
                ret[instance] = resources

        return ret


def _instances_total_count(hosts):
    ret = 0
    for h in hosts:
        ret += len(hosts[h])
    return ret


def prepare(shards, get_all):
    for k, v in shards.iteritems():
        if get_all or need(v):
            yield {'order': v['order'], 'value': v, 'key': k}


def need(shard):
    for host in shard['hosts'].itervalues():
        if not host['i'] or not host['spam'] or host['spam'] == '' or not all(host['instances']):
            return True
    return False


def get_tier_by_tags(tags):
    for tag in tags:
        # instance tag
        if 'a_tier_' in tag:
            return tag.replace('a_tier_', '')

        # instance tag without a tier
        if tag.startswith('a_'):
            continue

        # shard tag
        if any(w in tag for w in ('Tier', 'Maps', 'Tr', 'Jud', 'Kz', 'Ita', 'Eng', 'Rrg', 'Tur', 'Rus', 'OxygenExp')):
            return tag

    return None


def get_shard_from_tags(tags):
    for tag in tags:
        # instance tag
        if tag.startswith('a_shard_'):
            return tag.replace('a_shard_', '')

    return None


def get_spam_time(st):
    stime = time.localtime(st)
    return '{0}-{1}-{2}'.format(stime.tm_year, stime.tm_mon, stime.tm_mday)


def get_primus_group(shard):
    return str(PRIMUS_GROUPS.get(shard.split('-', 1)[0], 'other'))


def get_instance_order(instance):
    status = instance.get('status', '')
    if status == 'DONE':
        status = 'ZZZZ'
    return status, -instance.get('attempt', 0)


def get_status_text(status, attempt):
    if status == 'BUILD':
        return 'BUILD #{}'.format(attempt)
    if status == 'DOWNLOAD':
        return 'DOWNLOAD #{}'.format(attempt)
    return status


def parse_shard_statuses(s, yield_idle=False):
    try:
        obj = json.loads(s)
        shards = obj.get('shards', json.loads(s))
        for shard_id, shard_obj in shards.iteritems():
            status = shard_obj['status']
            if status in ['IDLE', 'NONE']:
                if yield_idle:
                    yield shard_id, status, 0
                else:
                    continue
            if status == 'BUILD':
                yield shard_id, status, shard_obj.get('build_attempt', 0)
            elif status == 'DOWNLOAD':
                yield shard_id, status, shard_obj.get('download_attempt', 0)
            else:
                yield shard_id, status, 0

    except Exception:
        logging.exception(s)


def parse_shard_status(s):
    try:
        obj = json.loads(s)
        shards = obj.get('shards', json.loads(s))
        for shard_id, shard_obj in shards.iteritems():
            status = shard_obj['status']
            if status in ['IDLE', 'NONE']:
                continue
            if status == 'BUILD':
                return shard_id, status, shard_obj.get('build_attempt', 0)
            if status == 'DOWNLOAD':
                return shard_id, status, shard_obj.get('download_attempt', 0)
            return shard_id, status, 0

    except Exception:
        logging.exception(s)

    return None, 'NONE', 0


def parse_shard_size(s):
    try:
        return int(s)
    except Exception:
        return 0


def build_shardtool_view(shard_statuses_inst):
    keys = ['BUILD', 'DOWNLOAD', 'DONE', 'FAILURE']
    keys += list(sorted(set(k for k in shard_statuses_inst.keys() if k not in keys)))

    return {
        'instances': {k: list(shard_statuses_inst.get(k, [])) for k in keys},
        'keys': keys,
    }


@singleton
def analyzer():
    return Analyzer()
