import logging
import logging.handlers
import os
import time
import sys


# noinspection PyPep8Naming
class ExtendedFormatter(logging.Formatter):
    def __init__(
            self,
            fmt='%(asctime)s - %(levelname)s - %(name)s - %(message)s',
            fmtName='%(name)s',
            fmtLevelno='%(levelno)d',
            fmtLevelname='%(levelname)s',
            fmtPathname='%(pathname)s',
            fmtFilename='%(filename)s',
            fmtModule='%(module)s',
            fmtLineno='%(lineno)d',
            fmtFuncName='%(funcName)s',
            fmtCreated='%(created)f',
            fmtAsctime='%(datetime)s,%(msecs)03d',
            fmtDatetime='%Y-%m-%d %H:%M:%S',
            fmtMsecs='%(msecs)d',
            fmtRelativeCreated='%(relativeCreated)d',
            fmtThread='%(thread)d',
            fmtThreadName='%(threadName)s',
            fmtProcess='%(process)d',
            fmtProcessName='%(processName)s',
            fmtHost='%(host)s',
            fmtMessage='%(message)s',
    ):
        super(ExtendedFormatter, self).__init__()

        self.fmt = None
        self.fmtAsctime = None
        self.fmtDatetime = None

        self.__dict__.update(locals())

    def formatTime(self, record, datefmt=None):
        ct = self.converter(record.created)
        datetime = time.strftime(datefmt if datefmt is not None else self.fmtDatetime, ct)
        return self.fmtAsctime % {
            'datetime': datetime,
            'msecs': record.msecs
        }

    def formatSimple(self, record, type_):
        default = '%%(%s)s' % (type_,)
        return getattr(self, 'fmt%s' % (type_[0].upper() + type_[1:],), default) % {
            type_: getattr(record, type_)
        }

    formatName = lambda self, record: self.formatSimple(record, 'name')
    formatLevelno = lambda self, record: self.formatSimple(record, 'levelno')
    formatLevelname = lambda self, record: self.formatSimple(record, 'levelname')
    formatPathname = lambda self, record: self.formatSimple(record, 'pathname')
    formatFilename = lambda self, record: self.formatSimple(record, 'filename')
    formatModule = lambda self, record: self.formatSimple(record, 'module')
    formatLineno = lambda self, record: self.formatSimple(record, 'lineno')
    formatFuncName = lambda self, record: self.formatSimple(record, 'funcName')
    formatCreated = lambda self, record: self.formatSimple(record, 'created')
    formatMsecs = lambda self, record: self.formatSimple(record, 'msecs')
    formatRelativeCreated = lambda self, record: self.formatSimple(record, 'relativeCreated')
    formatThread = lambda self, record: self.formatSimple(record, 'thread')
    formatThreadName = lambda self, record: self.formatSimple(record, 'threadName')
    formatProcess = lambda self, record: self.formatSimple(record, 'process')
    formatProcessName = lambda self, record: self.formatSimple(record, 'processName')
    formatHost = lambda self, record: self.formatSimple(record, 'host')
    formatMessage = lambda self, record: self.formatSimple(record, 'message')

    def prepare(self, record):
        record.message = record.getMessage()
        if '%(asctime)' in self.fmt:
            record.asctime = self.formatTime(record)
        if not hasattr(record, 'host'):  # Beware: hack for local and remote log records mix
            record.host = 'localhost'

        fmtData = {'asctime': record.asctime}

        for simpleItem in list(record.__dict__.keys()):
            if simpleItem in fmtData:
                continue
            if '%%(%s)' % simpleItem in self.fmt:
                try:
                    formatter = getattr(self, 'format%s' % (simpleItem[0].upper() + simpleItem[1:]))
                except AttributeError:
                    formatter = lambda record: self.formatSimple(record, simpleItem)
                    setattr(self, 'format%s' % (simpleItem[0].upper() + simpleItem[1:]), formatter)
                fmtData[simpleItem] = formatter(record)

        return fmtData

    def format(self, record):
        return self.fmt % self.prepare(record)


class ColoredFormatter(ExtendedFormatter):
    COLORS = dict(zip(
        ('BLACK', 'RED', 'GREEN', 'YELLOW', 'BLUE', 'MAGENTA', 'CYAN', 'WHITE'),
        [{'num': x} for x in range(30, 38)]
    ))

    # ALIASES = {
    #     'NORMAL': 'INFO',
    #     'UNKNOWN': 'NOTSET'
    # }

    # for key, values in six.iteritems(_levelNameAliases):
    #     for value in values:
    #         ALIASES[value] = key

    for color, info in list(COLORS.items()):
        COLORS[color]['normal'] = '\033[0;%dm' % info['num']
        COLORS[color]['bold'] = '\033[1;%dm' % info['num']

    RESET_SEQ = '\033[m'

    def __init__(self, **kwargs):
        # Defaults for color_map and Hungry levels
        color_map = {
            'levels': {
                # LEVEL:    (COLOR,    BOLD_OR_NOT)
                'PROTOCOL': ('WHITE', True),
                'DEBUG': ('BLACK', True),
                'INFO': ('GREEN', False),
                'WARNING': ('YELLOW', True),
                'ERROR': ('RED', False),
                'CRITICAL': ('RED', True),
            },
            'parts': {
                # PARTNAME: (COLOR,    BOLD_OR_NOT
                'asctime': ('BLACK', True),
                'name': ('CYAN', True),
                'host': ('BLACK', True),
            },
        }

        hungry_levels = {
            'CRITICAL': ('full',),  # Critical logs will have colorized full log message by default
            'ERROR': ('message', 'levelname'),  # Error logs will be colorized too (only message part)
        }

        self.useColors = kwargs.pop('useColors', True)

        for key, value in list(kwargs.items()):
            dict_ = None
            key2 = None
            if key.startswith('color'):
                dict_ = color_map['parts']
                key2 = key[len('color'):].lower()
            elif key.startswith('level'):
                dict_ = color_map['levels']
                key2 = key[len('level'):].upper()

            if dict_ is None:
                continue

            bold = value.endswith('+')
            value = value.rstrip('+')
            dict_[key2] = (value.upper(), bold)
            kwargs.pop(key)

        for levelName, childNames in kwargs.pop('hungryLevels', {}).items():
            hungry_levels[levelName.upper()] = childNames

        self.colorMap = color_map
        self.hungryLevels = hungry_levels

        ExtendedFormatter.__init__(self, **kwargs)

    def colorize(self, string, info):
        return ''.join((
            self.COLORS[info[0]]['normal' if not info[1] else 'bold'],
            string,
            self.RESET_SEQ,
        ))

    def format(self, record):
        if record.exc_info:
            # Cache the traceback text to avoid converting it multiple times
            # (it's constant anyway)
            if not record.exc_text:
                record.exc_text = self.formatException(record.exc_info)

        if record.exc_text:
            if not isinstance(record.msg, basestring):
                record.msg = str(record.msg)
            if not record.msg.endswith('\n'):
                record.msg += '\n'
            try:
                record.msg += record.exc_text
            except UnicodeError:
                # Sometimes filenames have non-ASCII chars, which can lead
                # to errors when s is Unicode and record.exc_text is str
                # See issue 8924
                record.msg += record.exc_text.decode(sys.getfilesystemencoding())

        if not self.useColors:
            return ExtendedFormatter.format(self, record)

        data = self.prepare(record)

        levelname = record.levelname
        level_color = self.colorMap['levels'].setdefault(levelname, None)
        if level_color is not None:
            level_hungry = self.hungryLevels.setdefault(levelname, ('levelname',))
            if 'full' in level_hungry:
                # Colorize full message and return immediately if we are
                # at the very very hungry level
                return self.colorize(self.fmt % data, level_color)
        else:
            level_hungry = ()

        # Colorize all specified parts
        for partName, part_color in list(self.colorMap['parts'].items()):
            if partName in level_hungry:
                part_color = level_color
            if partName not in data:
                continue
            data[partName] = self.colorize(data[partName], part_color)

        # Colorize all hungry parts by level color
        if level_color is not None:
            for partName in ['levelname'] + list(level_hungry):
                data[partName] = self.colorize(data[partName], level_color)

        return self.fmt % data


def ensure_dir(path):
    if not os.path.exists(path):
        os.makedirs(path)


def configure_logging(debug, logs_dir='./logs'):
    ensure_dir(logs_dir)

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    debug_handler = logging.handlers.RotatingFileHandler(logs_dir + '/debug.log', maxBytes=1024 ** 3, backupCount=10)
    debug_handler.setLevel(logging.DEBUG)
    configure_handler(debug_handler, use_colors=False)
    logger.addHandler(debug_handler)

    info_handler = logging.handlers.RotatingFileHandler(logs_dir + '/info.log', maxBytes=1024 ** 3, backupCount=10)
    info_handler.setLevel(logging.INFO)
    configure_handler(info_handler, use_colors=False)
    logger.addHandler(info_handler)

    if debug:
        console_handler = logging.StreamHandler()
        console_handler.setLevel(logging.DEBUG)
        configure_handler(console_handler, use_colors=True)
        logger.addHandler(console_handler)

    logging.getLogger('requests').setLevel(logging.CRITICAL)
    logging.getLogger('httpserver').setLevel(logging.CRITICAL)
    logging.getLogger('werkzeug').addHandler(debug_handler)
    logging.getLogger('werkzeug').setLevel(0)


def configure_handler(handler, use_colors):
    formatter = ColoredFormatter(
        useColors=use_colors,
        fmt='%(asctime)s %(levelname)s %(name)s  %(message)s',
        fmtName='[%(name)-7s]',
        fmtLevelname='[%(levelname)-1s]',
        fmtAsctime='%(datetime)s.%(msecs)003d',
        hungryLevels={
            'info': ['message'],
            'debug': ['message'],
            'protocol': ['message'],
            'warning': ['message'],
            'error': ['message', 'name']
        }
    )

    handler.setFormatter(formatter)
