# coding=utf-8

import re
from collections import defaultdict
import datetime as dt
import logging
import time
from datetime import datetime, timedelta
from functools import partial

import pymongo
import msgpack

from auto_updating_context import AutoUpdatingContext
from libraries.hardware import get_hosts_data, search_hosts
from libraries.online_state import InstanceState
from utils import singleton, shortname, logtime, run_daemon_loop, memoized
from state_obj import StateV3
from blinov import resolve
from yasm import stats_manager, ABSOLUTE

from mongo_params import ALL_HEARTBEAT_B_MONGODB, ALL_HEARTBEAT_C_MONGODB, HEARTBEAT_MONGODB


ISTATES_UPDATE_INTERVAL = 10
STATES_UPDATE_INTERVAL = 10
HOST_LIVENESS_TIMEOUT = 30  # minutes


class States(object):
    def __init__(self):
        self.client = get_mongo_client()
        self.db = self.client['heartbeat']

        self.log = logging.getLogger('states')

        self.skyinfo_cache = {}
        self.crashes_cache = [{}, {}, {}]
        self.oss_cache = {}
        self._states = {}
        self._istates = {}
        self.alive_hosts_cache = set()
        self.softinfo_cache = {}
        self.packages_cache = {}
        self.golovan_cache = {}
        self.iss_cache = {}
        self.bsconfig_iss_cache = dict(bsconfig_active=0, bsconfig_known=0, iss_active=0, iss_known=0)
        self._active_configurations = set()
        self.monitoring_cache = {}
        self.instance_state = InstanceState()

        self.funcs = [self.update_alive, self.update_crashes, self.update_istates, self.update_monitoring,
                      self.update_packages, self.update_skyinfo, self.update_softinfo, self.update_states,
                      self.update_bsconfig_iss]
        self.intervals = [300, 60, 10, 60, 300, 60, 300, 10, 60]

        self._context = AutoUpdatingContext()
        self._context.submit('alive', get_alive, default=set())
        self._context.submit('network', partial(get_network, 2), default={'I@ALL_SEARCH': {'dcs': {}}},
                             args=['alive'], callback=partial(self._network_to_yasm, 2), timeout=600)
        self._context.submit('network3', partial(get_network, 3), default={'I@ALL_SEARCH': {'dcs': {}}},
                             args=['alive'], callback=partial(self._network_to_yasm, 3), timeout=600)

    def _network_to_yasm(self, version, result):
        for tag, net_state in result['data'].iteritems():
            tag = re.sub(r'([^\w\-+=\.]+)', '_', tag).lower()
            yasm_state = net_state.setdefault('yasm', {})
            for dc, dc_state in net_state['dcs'].iteritems():
                yasm_dc = yasm_state.setdefault(dc, {})
                for net, counts in dc_state['counts'].iteritems():
                    prefix = 'network%d_%s_%s_%s' % (version, tag, dc, net)
                    yasm_dc[net] = prefix
                    stats_manager.setCounterValue('%s_probe' % prefix, counts['c'], ABSOLUTE)
                    stats_manager.setCounterValue('%s_score' % prefix, counts['p'], ABSOLUTE)

    def start_cachers(self):
        self._context.start()
        map(run_daemon_loop, self.funcs, self.intervals)

    def stop_cachers(self):
        self._context.stop()

    def warmup(self):
        for f in self.funcs:
            f()

    def states(self):
        return self._states

    def istates(self):
        return self._istates

    def alive(self):
        return self.alive_hosts_cache

    def crashes(self):
        return self.crashes_cache

    def skyinfo(self):
        return self.skyinfo_cache

    def oss(self):
        return self.oss_cache

    def softinfo(self):
        return self.softinfo_cache

    def golovan(self):
        return self.golovan_cache

    def iss(self):
        return self.iss_cache

    def monitoring(self):
        return self.monitoring_cache

    def active_configurations(self):
        return self._active_configurations

    @logtime
    def update_states(self):
        new_states = {}
        coll = self.db['shardstatev3']
        res = coll.find(
            {'last_update': {'$gt': dt.datetime.now() - dt.timedelta(seconds=STATES_UPDATE_INTERVAL * 3 if len(
                self.states()) else 3 * 60 * 60)}},
            {'host': 1, '_id': 0, 'state': 1, 'last_update': 1}
        )
        for r in res:
            host = shortname(r['host'])
            last_update = int(r['last_update'].strftime("%s"))
            if host not in new_states or new_states[host]['t'] < last_update:
                try:
                    new_states[host] = msgpack.loads(r['state'])
                    for k, v in new_states[host].iteritems():
                        new_states[host][k] = StateV3(v)
                    new_states[host]['t'] = last_update
                except KeyError:
                    self.log.info('Error loading report from %s', host)

        new_hosts = self.states().copy()
        new_hosts.update(new_states)
        for h in new_hosts.keys():
            if h not in self.alive():
                new_hosts.pop(h)
        self._states = new_hosts

    @logtime
    def update_istates(self):
        self.instance_state.update(self.alive(), self.db)
        self._istates = self.instance_state.ihosts

    def bsconfig_iss(self):
        return self.bsconfig_iss_cache

    @logtime
    def update_bsconfig_iss(self):
        runtime_hosts = frozenset(host['name'] for host in search_hosts())

        ret = {'bsc': {'a': set(), 'k': set()}, 'iss': {'a': set(), 'k': set()}}
        active_instances = {'bsconfig': 0, 'iss': 0}
        active_configurations = set()

        for host, data in self.istates().iteritems():
            for conf_name, conf_data in data.get('c', {}).iteritems():
                conf_is_active = conf_data['a']
                if conf_is_active:
                    active_configurations.add(conf_name)
                if host in runtime_hosts:
                    if conf_data.get('is_iss', 0) == 1:
                        ret['iss']['k'].add(conf_name)
                        if conf_is_active:
                            ret['iss']['a'].add(conf_name)
                    else:
                        ret['bsc']['k'].add(conf_name)
                        if conf_is_active:
                            ret['bsc']['a'].add(conf_name)
            for instance in data.get('i', {}).itervalues():
                conf = (instance.get('conf', '') or '').split('#')[-1]
                if conf in ret['bsc']['a']:
                    active_instances['bsconfig'] += 1
                elif conf in ret['iss']['a']:
                    active_instances['iss'] += 1

        self.bsconfig_iss_cache = dict(
            bsconfig_active=len(ret['bsc']['a']),
            bsconfig_known=len(ret['bsc']['k']),
            iss_active=len(ret['iss']['a']),
            iss_known=len(ret['iss']['k']),
            bsconfig_instances=active_instances['bsconfig'],
            iss_instances=active_instances['iss'],
        )

        self._active_configurations = active_configurations

    @logtime
    def update_alive(self):
        alive = set()
        for r in self.db['hostinfo'].find(
            {'last_update': {'$gt': dt.datetime.now() - dt.timedelta(minutes=HOST_LIVENESS_TIMEOUT)}},
            {'host': 1, '_id': 0}
        ):
            alive.add(shortname(r['host']))

        self.alive_hosts_cache = alive

    @logtime
    def update_crashes(self):
        coll = self.db['searchcrash']
        new_crashes = [{}, {}, {}]
        for cache_element, interval in [(0, 2), (1, 6), (2, 16)]:
            res = coll.aggregate([
                {
                    '$match': {
                        'generated': {
                            '$gt': dt.datetime.now() - dt.timedelta(minutes=interval)
                        }
                    }
                }, {
                    '$group': {
                        '_id': '$instance',
                        'total': {'$sum': 1}
                    }
                }
            ])
            for r in res:
                instance = shortname(r['_id'])
                new_crashes[cache_element].setdefault(instance, 0)
                new_crashes[cache_element][instance] += r['total']

        self.crashes_cache = new_crashes

    @logtime
    def update_skyinfo(self):
        new_skyinfo = {}
        new_oss = {}
        res = self.db['hostinfo'].find({'last_update': {'$gt': dt.datetime.now() - dt.timedelta(minutes=30)}},
                                       {'host': 1, 'os': 1, '_id': 0, 'skynet.srvmngr.services': 1})
        for r in res:
            host = shortname(r['host'])
            if host not in self.alive():
                continue
            new_skyinfo[host] = r.get('skynet', {}).get('srvmngr', {}).get('services', {})
            new_oss[host] = {'os': str(r.get('os', {}).get('name', 'None')),
                             'ver': str(r.get('os', {}).get('version', 'None'))}
        self.skyinfo_cache = new_skyinfo
        self.oss_cache = new_oss

    @logtime
    def update_packages(self):
        packages = {}
        for doc in self.db.softinfo.find({}, {'host': 1, '_id': 0, 'dpkg': 1}):
            host = shortname(doc['host'])
            if host in self.alive():
                packages[host] = doc['dpkg']
        self.packages_cache = packages

    def packages(self):
        return self.packages_cache

    @logtime
    def update_softinfo(self):
        d = {}
        g = {}
        i = {}

        for doc in self.db.softinfo.find({}, {'host': 1, '_id': 0, 'tejblum': 1, 'golovan': 1, 'iss': 1}):
            host = shortname(doc['host'])
            d[host] = {'tejblum': doc['tejblum']}
            g[host] = doc.get('golovan', 'unknown')
            if g[host] == '':
                g[host] = 'unknown'
            i[host] = doc.get('iss', 'unknown')
            if i[host] == '':
                i[host] = 'unknown'

        self.softinfo_cache = d
        self.golovan_cache = g
        self.iss_cache = i

    @logtime
    def update_monitoring(self):
        hosts = {'cqueue': {}, 'cqudp': {}}
        copier_hosts = {}

        for r in self.db['monitoring'].find({'last_update': {'$gt': dt.datetime.now() - dt.timedelta(minutes=30)}},
                                            {'host': 1, 'services': 1, '_id': 0}):
            host = shortname(r['host'])
            if host not in self.alive():
                continue
            for service in hosts.iterkeys():
                try:
                    hosts[service][host] = bool(r['services'][service]['ping'])
                except KeyError:
                    pass
                try:
                    copier_hosts[host] = str(r['services']['copier']['report']['download']['result']) == 'ok'
                except KeyError:
                    pass
            self.monitoring_cache = {'cqueue': hosts['cqueue'], 'cqudp': hosts['cqudp'], 'copier': copier_hosts}

    def file_sizes(self, epoch, file_name):
        file_name = file_name.replace('.', u'\uff0e')

        return [(r['files'][file_name], r['zone']) for r in self.db['shardsize'].find(
            {'epoch': epoch, 'files.' + file_name: {'$exists': True}},
            {'files.' + file_name: 1, 'zone': 1, '_id': 0}
        )]

    def file_sizes_histogram(self, epoch1, epoch2, filename):
        dd = (defaultdict(list), defaultdict(list))
        for i in [0, 1]:
            for x in self.file_sizes([epoch1, epoch2][i], filename):
                dd[i][x[1]].append(x[0])

        return {z: histogram2(dd[0][z], dd[1][z], 100) for z in set(dd[0].keys() + dd[1].keys())}

    def doc_count(self, file_name):
        file_name = file_name.replace('.', u'\uff0e')

        ret = {}

        res = self.db['shardsize'].aggregate([
            {"$match": {"shard": {"$regex": "^primus|^lp"}, "epoch": {"$gt": int(time.time() - 3600*24*90)}}},
            {"$group": {"_id": {"zone": "$zone", "epoch": "$epoch"}, "total": {"$sum": u"$files.{0}".format(file_name)}}}
        ])
        for r in res:
            zone = r['_id']['zone']
            ret.setdefault(zone, {})

            # ironpeter@ told that ui16 and we should devide by 2
            ret[zone][r['_id']['epoch']] = r['total'] / 2

        return ret

    def _network_get_expr(self, expr=None):
        return 'I@ALL_SEARCH' if expr is None else expr

    def _network_get_version(self, version):
        return int(version) if version else 2

    def _network_get_type(self, proto, version=None):
        network_name, _, protocol_name = proto.partition('_')
        if network_name not in ('bb4', 'bb6', 'fb6'):
            raise ValueError('Unknown network {0} given'.format(network_name))
        if protocol_name not in ('', 'icmp', 'udp', 'tcp'):
            raise ValueError('Unknown protocol {0} given'.format(protocol_name))
        if self._network_get_version(version) >= 3:
            return '{0}_score'.format(proto) if '_' in proto else proto
        else:
            return proto

    def _network_get_data(self, version=None):
        key = 'network3' if self._network_get_version(version) >= 3 else 'network'
        state = self._context[key]
        if datetime.now() - state['timestamp'] > timedelta(seconds=300):
            self.log.error('State for %s is too old', key)
            if key == 'network':
                raise RuntimeError('Too old network state')

            return {'I@ALL_SEARCH': {'dcs': {}}}
        else:
            return state['data']

    def _network_get_view(self, expr=None, version=None):
        return self._network_get_data(version).get(self._network_get_expr(expr))

    def _network_get_filter(self, expr=None, period_in_minutes=2, version=None, proto=None):
        expr = self._network_get_expr(expr)
        filter_expr = {
            'generated': {
                '$gt': dt.datetime.now() - dt.timedelta(minutes=period_in_minutes)
            }
        }
        if proto is not None:
            filter_expr[proto] = {'$exists': 1}
        if self._network_get_version(version) >= 3 and expr in self._network_get_data(version):
            # show probes only in given view
            filter_expr['tags'] = expr
        return filter_expr

    def network_views(self, version=None):
        return sorted(self._network_get_data(version))

    def network(self, json='no', period_in_minutes=2, expr=None, version=None):
        network_data = self._network_get_view(expr, version)
        network_data = {'dcs': {}} if network_data is None else network_data

        if json in ['da', 'yes']:
            ret = {'type': 'pess'}
            for d, v in network_data['dcs'].iteritems():
                ret[d] = {}
                for net in v['counts'].iterkeys():
                    ret[d].setdefault(net, {})
                    ret[d][net]['total'] = v['counts'][net]['c']
                    ret[d][net]['success'] = v['counts'][net]['p']
            return ret
        else:
            return network_data

    def network_ab(self, network_type=None, period_in_minutes=2, expr=None, version=None):
        hosts_map = get_hosts_data()

        known_hosts = None
        if self._network_get_view(expr, version) is None:
            known_hosts = set(resolve(expr)[0])

        coll = get_pinger_coll(version, db=self.db)

        type_list = [network_type] if network_type is not None else ['bb6']
        network = {type_name: {} for type_name in type_list}
        for net in type_list:
            res = coll.find(
                self._network_get_filter(expr, period_in_minutes, version),
                {'source': 1, '_id': 0, 'target': 1, net: 1}
            )

            if net == 'bb4':
                main_dcs = ['ugrb', 'iva', 'folA', 'folB']
            else:
                main_dcs = ['myt', 'ugrb', 'iva', 'folA', 'folB', 'man', 'sas', 'vla']

            for doc in res:
                a = shortname(doc['source'])
                b = shortname(doc['target'])

                if known_hosts is not None and (a not in known_hosts or b not in known_hosts):
                    continue

                if a not in self.alive() or b not in self.alive():
                    continue

                try:
                    da, la, sa = hosts_map[a]
                    da = fix_dc(da, la)
                except KeyError:
                    continue
                try:
                    db, lb, sb = hosts_map[b]
                    db = fix_dc(db, lb)
                except KeyError:
                    continue

                p = get_net_value(doc, net)
                if len(da) < 1 or da == 'unknown' or len(db) == 0 or db == 'unknown' or p < 0 or p is None:
                    continue

                if da not in main_dcs or db not in main_dcs:
                    continue

                if (a not in self.oss()) or (net == 'bb4' and self.oss()[a]['os'] == 'FreeBSD'):
                    continue

                network[net].setdefault(da, {})
                network[net][da].setdefault(db, {'c': 0, 'p': 0})
                network[net][da][db]['c'] += 1
                network[net][da][db]['p'] += p

        return network

    def network_ab_(self, network_type='bb6', dc_a=None, dc_b=None, line_a=None, line_b=None,
                    depth=None, period_in_minutes=2, json='no', expr=None, version=None):
        network_type = self._network_get_type(network_type, version)
        expr = self._network_get_expr(expr)
        depth = int(depth or 0)
        hosts_map = get_hosts_data()
        net = {}

        known_hosts = None
        if self._network_get_view(expr, version) is None:
            known_hosts = set(resolve(expr)[0])

        coll = get_pinger_coll(version, db=self.db)
        filter_expr = self._network_get_filter(expr, period_in_minutes, version)
        filter_expr[network_type] = {'$exists': 1}
        res = coll.find(filter_expr, {'_id': 0, 'source': 1, 'target': 1, network_type: 1})
        alive_hosts = self.alive()
        oss_hosts = self.oss()

        def init_pairs(root, key_a, key_b):
            root.setdefault(key_a, {})
            root.setdefault(key_b, {})
            root[key_a].setdefault(key_a, {'bad': list(), 'cnt': 0, 'success': 0, 'net': {}})
            root[key_a].setdefault(key_b, {'bad': list(), 'cnt': 0, 'success': 0, 'net': {}})
            if key_a != key_b:
                root[key_b].setdefault(key_a, {'bad': list(), 'cnt': 0, 'success': 0, 'net': {}})
                root[key_b].setdefault(key_b, {'bad': list(), 'cnt': 0, 'success': 0, 'net': {}})

        for doc in res:
            a = shortname(doc['source'])
            b = shortname(doc['target'])

            if known_hosts is not None and (a not in known_hosts or b not in known_hosts):
                continue

            if a not in oss_hosts or oss_hosts[a]['os'] == 'FreeBSD':
                continue

            if b not in alive_hosts:
                continue

            try:
                da, la, sa = hosts_map[a]
                da = fix_dc(da, la)
                db, lb, sb = hosts_map[b]
                db = fix_dc(db, lb)
            except KeyError:
                continue

            if da == 'unknown' or db == 'unknown' or len(da) == 0 or len(db) == 0:
                continue

            if (dc_a is not None and dc_a != da) or (dc_b is not None and dc_b != db):
                continue

            if (line_a is not None and line_a != la) or (line_b is not None and line_b != lb):
                continue

            p = get_net_value(doc, network_type, t='as is')
            init_pairs(net, da, db)
            dc_net = net[da][db]
            dc_net['cnt'] += 1
            dc_net['success'] += max(p, 0)
            if da == db and depth >= 1:
                init_pairs(dc_net['net'], la, lb)
                line_net = dc_net['net'][la][lb]
                line_net['cnt'] += 1
                line_net['success'] += max(p, 0)
                if la == lb and depth >= 2:
                    init_pairs(line_net['net'], sa, sb)
                    switch_net = line_net['net'][sa][sb]
                    switch_net['cnt'] += 1
                    switch_net['success'] += max(p, 0)

            if p < 1 and json not in ['yes', 'da']:
                net[da][db]['bad'].append('{0}({1}) -> {2}({3}): {4}'.format(a, sa, b, sb, p))

        return net

    def network_(self, proto='bb4', stat_type='switch', period_in_minutes=2, pess='opt', alive='alive', expr=None, version=None):
        proto = self._network_get_type(proto, version)
        expr = self._network_get_expr(expr)
        all_search_hosts = set(resolve(expr)[0])
        hosts_map = get_hosts_data()
        net = {}

        coll = get_pinger_coll(version, db=self.db)

        if alive == 'pinger':
            all_pinger_hosts = set()
            res = coll.find(self._network_get_filter(expr, period_in_minutes, version)).distinct('source')
            for doc in res:
                all_pinger_hosts.add(shortname(doc))

        filter_expr = self._network_get_filter(expr, period_in_minutes, version)
        filter_expr[proto] = {'$exists': 1}
        res = coll.find(filter_expr, {'_id': 0, 'source': 1, 'target': 1, proto: 1})

        for doc in res:
            a = shortname(doc['source'])
            b = shortname(doc['target'])

            if a not in all_search_hosts or b not in all_search_hosts:
                continue

            if alive == 'pinger':
                if a not in all_pinger_hosts or b not in all_pinger_hosts:
                    continue

            if b not in self.alive():
                continue

            try:
                dc, line, switch = hosts_map[b]
                dc = fix_dc(dc, line)
            except KeyError:
                continue

            key = {'dc': dc, 'line': line, 'switch': switch}[stat_type]
            net.setdefault(key, {'bad': list(), 'cnt': 0})
            net[key]['cnt'] += 1

            p = get_net_value(doc, proto, pess)

            if 1 > p >= 0:
                net[key]['bad'].append('{0} -> {1}: {2}'.format(a, b, p))

        return net


def get_alive(timestamp):
    new_timestamp = datetime.now() - timedelta(seconds=5)  # overlap to guarantee

    # hosts that have not been reporting new statuses for 30 minutes considered to be dead
    new_alive = set()
    res = get_coll('hostinfo').find(
        {'last_update': {'$gt': dt.datetime.now() - dt.timedelta(minutes=30)}},
        {'host': 1, '_id': 0}
    )
    for r in res:
        new_alive.add(r['host'])

    return {'timestamp': new_timestamp, 'data': new_alive}


def fill_network_map(host, net, d, l, s, p, network=None):

    def default_counts():
        return {
            'bb4': {'c': 0, 'p': 0, 'h': set()},
            'bb6': {'c': 0, 'p': 0, 'h': set()},
            'fb6': {'c': 0, 'p': 0, 'h': set()}
        }

    if network is None:
        network = {'dcs': {}, 'counts': default_counts()}

    network['counts'][net]['c'] += 1
    network['counts'][net]['p'] += p

    dc = network['dcs'].setdefault(d, {'lines': {}, 'counts': default_counts()})
    dc['counts'][net]['c'] += 1
    dc['counts'][net]['p'] += p
    if p != 1:
        dc['counts'][net]['h'].add(host)

    line = dc['lines'].setdefault(l, {'switches': {}, 'counts': default_counts()})
    line['counts'][net]['c'] += 1
    line['counts'][net]['p'] += p
    if p != 1:
        line['counts'][net]['h'].add(host)

    switch = line['switches'].setdefault(s, {'counts': default_counts()})
    switch['counts'][net]['c'] += 1
    switch['counts'][net]['p'] += p
    if p != 1:
        switch['counts'][net]['h'].add(host)

    return network


def get_network(version, timestamp, alive):
    period_in_minutes = 2

    views = {}
    hosts_map = get_hosts_data()
    coll = get_pinger_coll(version)

    if version >= 3:
        tags_list = coll.distinct("tags")
    else:
        tags_list = ['I@ALL_SEARCH']

    for tag in tags_list:
        for net in ['bb4', 'bb6', 'fb6']:
            match_expr = {
                'generated': {'$gt': dt.datetime.now() - dt.timedelta(minutes=period_in_minutes)},
                net: {'$exists': 1}
            }
            if version >= 3:
                match_expr['tags'] = tag
            res = coll.aggregate([
                {"$match": match_expr},
                {"$group": {"_id": "$target", net: {"$max": "$" + net}}}
            ])

            for doc in res:
                host = doc['_id']
                if host not in alive:
                    continue
                host = shortname(host)

                try:
                    d, l, s = hosts_map[host]
                except KeyError:
                    continue
                p = get_net_value(doc, net)

                if d == 'unknown' or len(d) == 0 or p < 0:
                    continue

                views[tag] = fill_network_map(host, net, d, l, s, p, views.get(tag))

    return {'timestamp': datetime.now(), 'data': views}


@singleton
def get_mongo_client():
    return pymongo.MongoClient(
        HEARTBEAT_MONGODB.uri,
        replicaSet=HEARTBEAT_MONGODB.replicaset,
        read_preference=HEARTBEAT_MONGODB.read_preference,
        localThresholdMS=30000,
    )


def get_db():
    return get_mongo_client()['heartbeat']


def get_coll(name):
    return get_db()[name]


def histogram2(d1, d2, n):
    d = d1 + d2
    a = min(d)
    b = max(d)
    r1 = [0] * n
    r2 = [0] * n

    if a == b:
        r1[0] = n
        r2[0] = n
        return r1, r2, a, b, 0

    for d in d1:
        r1[int((d - a) / float(b - a) * (n - 1))] += 1
    for d in d2:
        r2[int((d - a) / float(b - a) * (n - 1))] += 1

    return r1, r2, a, b, (b - a) / n


def fix_dc(dc, line):
    if dc == 'fol':
        if line in ('fol-1', 'fol-2', 'fol-3', 'fol-7'):
            return 'folA'
        else:
            return 'folB'
    return dc


def get_net_value(doc, net, t='pess'):
    if net not in doc:
        return None

    value = doc[net]
    if value < 0:
        return value

    if t == 'pess':
        return 1 if value >= 1. else 0
    elif t == 'opt':
        return 1 if value >= 0.1 else 0
    else:
        return value


def get_pinger_coll(version=None, db=None):
    db = db if db is not None else get_db()
    version = int(version) if version else 2

    if version >= 3:
        return get_heartbeat_b_client()['heartbeat']['pinger3']

    return get_heartbeat_c_client()['heartbeat']['pinger2']


@memoized
def get_heartbeat_b_client():
    return pymongo.MongoClient(
        ALL_HEARTBEAT_B_MONGODB.uri,
        replicaSet=ALL_HEARTBEAT_B_MONGODB.replicaset,
        read_preference=ALL_HEARTBEAT_B_MONGODB.read_preference,
        localThresholdMS=30000,
        socketTimeoutMS=5000,
        waitQueueTimeoutMS=5000
    )


@memoized
def get_heartbeat_c_client():
    return pymongo.MongoClient(
        ALL_HEARTBEAT_C_MONGODB.uri,
        replicaset=ALL_HEARTBEAT_C_MONGODB.replicaset,
        read_preference=ALL_HEARTBEAT_C_MONGODB.read_preference,
        localThresholdMS=30000,
        socketTimeoutMS=5000,
        waitQueueTimeoutMS=5000
    )


@singleton
def states():
    return States()


if __name__ == '__main__':
    import pprint

    pprint.pprint(states().network_())
