import logging

import os
import socket
import tornado
import tornado.ioloop
import tornado.web
from tornado.web import url

import cms
import handlers as h
from states import states
from utils import run_daemon_loop
from workers import LimitedPool


class Web(tornado.web.Application):
    def __init__(self, port, warmup):
        cwd = os.getcwd()
        settings = {
            'template_path': os.path.join(cwd, 'templates'),
            'static_path': os.path.join(cwd, 'static'),
            'backend_hostname': socket.gethostname(),
        }

        tornado.web.Application.__init__(self, _handlers, **settings)

        self.port = port
        self.needs_warmup = warmup
        self.log = logging.getLogger('web_app')
        self.pool = LimitedPool(max_workers=64)
        self.http_server = None

    def start(self):
        if self.needs_warmup:
            self._warmup()

        self._start_cachers()

        self.log.info('starting web application on %s', self.port)
        self.http_server = self.listen(self.port)

        tornado.ioloop.IOLoop.instance().start()

    def _warmup(self):
        self.log.info('starting warmup')

        states().warmup()
        import libraries.hardware as hardware
        hardware.load_singletons()
        self.log.info('%s hosts', len(hardware.all_hosts()))

    def _start_cachers(self):
        self.log.info('starting cachers')

        states().start_cachers()

        run_daemon_loop(cms.update_cache, 120)


_handlers = [
    url('/', h.SearchHandler),
    url('/search', h.SearchHandler),
    url('/availability', h.AvailabilityHandler),
    url('/search_suggest', h.SuggestHandler),
    url('/panel', h.PannelHandler),
    url('/fpanel', h.FramedPannelHandler),
    url('/fcpanel', h.FramedCPanelHandler),
    url('/shardsize', h.ShardSizeHandler),
    url('/doccount', h.DocCountHandler),
    url('/outliers/history', h.OutliersHistoryHandler),
    url('/outliers$', h.OutliersChooseLocationHandler),
    url('/outliers/.{3}', h.OutliersHandler),
    url('/outliers/.{3}/framed', h.OutliersFramedHandler),
    url('/outliers/api_/.{3}', h.OutliersPlotJsonHandler),
    url('/waitinfo', h.WaitInfoHandler),
    url('/network_views', h.NetworkViewsHandler, name='network_views'),
    url('/network', h.NetworkHandler, name='network'),
    url('/network/framed', h.FramedNetworkHandler, dict(view_name='network'),
        name='framed_network'),
    url('/abnetwork', h.NetworkABHandler, name='abnetwork'),
    url('/abnetwork/framed', h.FramedNetworkHandler, dict(view_name='abnetwork'),
        name='framed_abnetwork'),
    url('/fabnetwork', tornado.web.RedirectHandler, dict(url='/abnetwork/framed')),
    url('/abnetwork_', h.NetworkAB_Handler, name='abnetwork_'),
    url('/abnetwork_/framed', h.FramedNetworkHandler, dict(view_name='abnetwork_'),
        name='framed_abnetwork_'),
    url('/pnetwork', h.NetworkPanelHandler, name='pnetwork'),
    url('/pnetwork/framed', h.FramedNetworkHandler, dict(view_name='pnetwork'),
        name='framed_pnetwork'),
    url('/fnetwork', tornado.web.RedirectHandler, dict(url='/pnetwork/framed')),
    url('/network_', h.Network_Handler, name='network_'),
    url('/network_/framed', h.FramedNetworkHandler, dict(view_name='network_'),
        name='framed_network_'),
    url('/resolve', h.ResolveHandler),
    url('/ping', h.PingHandler),
    url('/status', h.StatusHandler),
    url('/stats', h.StatsHandler),
    url('/gencfg', h.GencfgCommits),
    url('/gencfg/json', h.GencfgCommitsJson),
    url('/api/v1/tasks', h.SuspensionHttpApi),
    url('/api/v1/tasks/.*', h.SuspensionHttpApi2),
    url('/api/v1/instances/', h.SearchJsonHandler),
    url('/api/v1/hosts/.*', h.HostsApiHandler),
    url('/api/v1/groups/.*', h.GroupsApiHandler),
    url('/api/v1/hardware/lines/?', h.HardwareLinesApiHandler),
    url('/api/v1/hardware/lines/.+', h.HardwareLineApiHandler),
]
