# -*- coding: utf-8 -*-

import itertools
import threading
import math

ABSOLUTE = 'axxx'
DELTA = 'summ'


class Counter(object):
    def __init__(self, aggregation_method):
        self.value = 0
        self._lock = threading.Lock()
        self.aggregation_method = aggregation_method

    def set(self, value):
        with self._lock:
            self.value = value

    def add(self, value):
        with self._lock:
            self.value += value

    def getYasmFormat(self):
        return "count_" + self.aggregation_method, self.value


class Histogram(object):
    BASE = 1.5
    MIN_LOG = -50
    MAX_LOG = 50
    _BORDERS = [0.0] + [BASE ** x for x in xrange(MIN_LOG, MAX_LOG + 1)]

    def __init__(self):
        self._lock = threading.Lock()
        self._buckets = [0] * len(self._BORDERS)

    def getDifferenceFrom(self, other_hist):
        hist = Histogram()
        for idx, (left, right) in enumerate(itertools.izip(self._buckets, other_hist._buckets)):
            hist._buckets[idx] = max(left - right, 0)
        return hist

    def getPercentile(self, percentile):
        with self._lock:
            items_count = sum(self._buckets)
            items_required = items_count * percentile
            items_so_far = 0
            pct_bucket = 0
            for idx, value in enumerate(self._buckets):
                pct_bucket = idx
                items_so_far += value
                if items_so_far >= items_required:
                    break
        return (self._BORDERS[pct_bucket] + self._BORDERS[min(pct_bucket + 1, len(self._BORDERS)-1)]) / 2.0

    def add(self, value):
        if value == 0:
            with self._lock:
                self._buckets[0] += 1
            return
        offset = math.floor(max(self.MIN_LOG - 1, min(self.MAX_LOG, math.log(value, self.BASE))) - self.MIN_LOG + 1)
        with self._lock:
            self._buckets[int(offset)] += 1

    def getYasmFormat(self):
        first_nonzero = next((idx for idx, value in enumerate(self._buckets) if value > 0), 0)
        # actually next after the last non-zero
        last_nonzero = len(self._buckets) - next((idx for idx, value in enumerate(reversed(self._buckets)) if value > 0), 0)
        borders = self._BORDERS[first_nonzero:last_nonzero]
        values = self._buckets[first_nonzero:last_nonzero]
        return "hgram_dhhh", [[edge, count] for edge, count in itertools.izip(borders, values)]


class StatsManager(object):
    def __init__(self):
        self._counters = {}
        self._histograms = {}
        self._lock = threading.Lock()

    def reset(self):
        with self._lock:
            self._counters = {}
            self._histograms = {}

    def incrementCounter(self, key, value=1, aggregation=DELTA):
        self._get_counter(key, aggregation).add(value)

    def decrementCounter(self, key, value=1, aggregation=DELTA):
        self._get_counter(key, aggregation).add(-value)

    def setCounterValue(self, key, value, aggregation=DELTA):
        self._get_counter(key, aggregation).set(value)

    def getCounterValue(self, key):
        if key in self._counters:
            return self._counters[key].value
        return 0

    def addSample(self, key, value):
        self._get_histogram(key).add(value)

    def getYasmSignals(self):
        metrics_list = []
        for key, meter in self._iterate_metrics():
            suffix, value = meter.getYasmFormat()
            metrics_list.append(["{0}_{1}".format(key, suffix), value])
        return metrics_list

    def _iterate_metrics(self):
        return itertools.chain.from_iterable([
            # iteritems is not thread-safe
            self._counters.items(),
            self._histograms.items()
        ])

    def _get_counter(self, key, aggregation):
        with self._lock:
            if key not in self._counters:
                self._counters[key] = Counter(aggregation)
        counter = self._counters[key]
        if counter.aggregation_method != aggregation:
            raise ValueError("Cannot change aggregation method from {0} to {1}".format(counter.aggregation_method, aggregation))
        return counter

    def _get_histogram(self, key):
        with self._lock:
            if key not in self._histograms:
                self._histograms[key] = Histogram()
        return self._histograms[key]


# global stats manager
stats_manager = StatsManager()
