import os
import datetime as dt
import gevent
import logging
import pymongo
from libraries.online_state import InstanceState
from libraries.topology import load_tags, get_latest_success_commit
from libraries.topology.searcher_lookup import load_full_trunk
from libraries.topology.groups import get_gencfg_trunk_commits
from libraries.topology.staff import load_group_persons_map
from libraries.topology.slbs import load_slbs
from libraries.topology.hbf_macroses import load_hbf_macroses
from libraries.topology.host_info import load_hosts_to_groups, load_hosts_to_hardware
from libraries.utils import shortname, singleton

from mongo_params import HEARTBEAT_MONGODB


PATH_TO_DNS_BUNDLE = 'dns_bundle.json'
PATH_TO_INSTALLED_FLAG = '_installed_.flag'


class DeadMongo(Exception):
    message = 'mongo is dead'


class Updater(object):
    def __init__(self):
        self._active = False
        self._last_update = dt.datetime.min
        self._sleep = 120  # seconds
        self._db = None
        self._alive_hosts_cache = set()
        self._instance_state = InstanceState()
        self._tags = {}
        self._gencfg_trunk_commits = set()
        self._newest_commit = None

        self._gencfg_trunk = {}
        self._gencfg_dns_json = None
        self._gencfg_slbs = None
        self._gencfg_hbf_macroses = None
        self._gencfg_hosts_to_groups = None
        self._gencfg_hosts_to_hardware = None
        self._trunk_revision = None

        self._staff_group_person_map = {}

        self._update_tags_only = False

    @property
    def alive(self):
        return self._alive_hosts_cache

    @property
    def ready(self):
        return self._last_update > dt.datetime.min

    @property
    def since_last_update(self):
        return dt.datetime.now() - self._last_update

    @staticmethod
    def _file_age(path_to_file):
        dns_modified_time = dt.datetime.min
        if os.path.exists(path_to_file):
            dns_modified_time = dt.datetime.fromtimestamp(os.path.getmtime(path_to_file))
        return dt.datetime.now() - dns_modified_time

    @property
    def dns_age(self):
        return self._file_age(PATH_TO_DNS_BUNDLE)

    @property
    def snapshot_age(self):
        return self._file_age(PATH_TO_INSTALLED_FLAG)

    @property
    def groups(self):
        return self._instance_state.groups_state

    @property
    def hosts(self):
        return self._instance_state.ihosts

    @property
    def tags(self):
        return self._tags

    @property
    def gencfg_trunk(self):
        return self._gencfg_trunk

    @property
    def gencfg_trunk_revision(self):
        return self._trunk_revision

    @property
    def gencfg_trunk_newest_commit(self):
        return max(self._gencfg_trunk_commits)

    @property
    def gencfg_trunk_oldest_commit(self):
        return min(self._gencfg_trunk_commits)

    @property
    def gencfg_trunk_commits(self):
        return self._gencfg_trunk_commits

    @property
    def staff_group_person_map(self):
        return self._staff_group_person_map

    @property
    def gencfg_dns(self):
        return self._gencfg_dns_json

    @property
    def gencfg_slbs(self):
        return self._gencfg_slbs

    @property
    def gencfg_hbf_macroses(self):
        return self._gencfg_hbf_macroses

    @property
    def gencfg_hosts_to_groups(self):
        return self._gencfg_hosts_to_groups

    @property
    def gencfg_hosts_to_hardware(self):
        return self._gencfg_hosts_to_hardware

    def _get_groups(self):
        self._instance_state.update(self.alive, self._db)

    def _get_tags(self):
        self._tags = load_tags()

    def _get_alive(self):
        alive = set()
        cursor = self._db['hostinfo'].find(
            {'last_update': {'$gt': dt.datetime.now() - HOST_LIVENESS_TIMEOUT}},
            {'host': 1, '_id': 0}
        )
        for r in cursor:
            alive.add(shortname(r['host']))
        self._alive_hosts_cache = alive

    def _recalc_alive(self):
        self._instance_state.groups_state.recalc_alive(sleep_method=gevent.sleep)

    def _load_trunk(self):
        commit = int(get_latest_success_commit())
        if commit != self._newest_commit:
            (
                self._gencfg_trunk,
                self._newest_commit,
            ) = (
                load_full_trunk(commit, gevent.sleep),
                commit,
            )
        self._gencfg_trunk_commits = get_gencfg_trunk_commits()
        self._gencfg_slbs = load_slbs(commit)
        self._gencfg_hbf_macroses = load_hbf_macroses()

        self._gencfg_hosts_to_groups = {
            x['hostname']: x['data']
            for x in load_hosts_to_groups(commit)
        }
        self._gencfg_hosts_to_hardware = {
            x['hostname']: x['data']
            for x in load_hosts_to_hardware(commit)
        }

        self._trunk_revision = commit

    def _load_staff(self):
        self._staff_group_person_map = load_group_persons_map()

    def _load_dns(self):
        path_to_bundle = PATH_TO_DNS_BUNDLE
        if os.path.exists(path_to_bundle):
            with open(path_to_bundle) as f:
                self._gencfg_dns_json = f.read()
        else:
            logging.warning('dns_bundle not found')

    def _update_all(self):
        self._get_tags()
        logging.info('done tags')
        self._get_alive()
        logging.info('done alive')
        self._get_groups()
        logging.info('done groups')
        self._load_trunk()
        logging.info('done load_trunk')
        self._recalc_alive()
        logging.info('done recalc_alive')
        self._load_staff()
        logging.info('done load staff')
        self._load_dns()
        logging.info('done load dns')
        self._last_update = dt.datetime.now()

    def _update_tags(self):
        self._get_tags()
        logging.info('done tags')
        self._load_staff()
        logging.info('done load staff')
        self._last_update = dt.datetime.now()

    def start(self):
        self._db = get_mongo_client()['heartbeat']
        self._active = True
        while self._active:
            try:
                with gevent.Timeout(20 * 60, DeadMongo):
                    if self._update_tags_only:
                        self._update_tags()
                    else:
                        self._update_all()
                gevent.sleep(self._sleep)
            except DeadMongo:
                logging.error('mongo died')
            except Exception as ex:
                logging.exception(ex)
                gevent.sleep(self._sleep)

    def stop(self):
        self._active = False

    def update_tags_only(self):
        logging.warning("will update tags only!")
        self._update_tags_only = True


@singleton
def get_mongo_client():
    return pymongo.MongoReplicaSetClient(
        HEARTBEAT_MONGODB.uri,
        replicaSet=HEARTBEAT_MONGODB.replicaset,
        localThresholdMS=30000,
        connectTimeoutMS=5000,
        read_preference=HEARTBEAT_MONGODB.read_preference
    )


HOST_LIVENESS_TIMEOUT = dt.timedelta(minutes=30)


updater = Updater()
