import json
import collections

import flask
import gevent

from web import app

from libraries.topology import get_group_by_version, load_instances, load_card, get_all_topology_groups
from libraries.topology.instances import get_instances_by_version
from libraries.topology.bsconfig import json_to_bsconfig
from libraries.topology.searcher_lookup import (
    canonize_ipv6, load_group_trunk, ips_of_hostname, hostname_of_ip, load_instances_tags,
    group_of_hostname,
)
from libraries.topology.utils import tag_to_version, commit_to_trunk, is_trunk, version_to_tag, trunk_to_commit
from libraries.topology.tags import get_tag_by_commit
import libraries.hardware as hardware
from libraries.topology.cpumodels import get_cpumodels

from updaters import updater


@app.route('/trunk/description')
def api_get_trunk_description():
    revision = updater.gencfg_trunk_revision
    data = {
        'description': '(trunk, rev. {})'.format(revision),
        'revision': revision,
    }
    return flask.jsonify(data)


@app.route('/tags/<tag>/groups/<group>')
def api_get_tags_groups(tag, group):
    version = tag_to_version(tag)
    data = get_group_by_version(group, version)
    if data:
        res = {key: value for key, value in data.items() if key in {'group', 'master', 'owners'}}
        res['hosts'] = data['hosts'].keys()
        return flask.jsonify(res)
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/groups/<group>')
def api_get_trunk_groups(group):
    data = updater.gencfg_trunk.get(group)
    if data:
        data = data.get()
        res = {key: value for key, value in data.items() if key in {'group', 'master', 'owners'}}
        res['hosts'] = data['hosts']
        return flask.jsonify(res)
    return flask.Response('group does not exist', status=404)


def _get_trunk_commit_group(commit, group):
    if commit == updater.gencfg_trunk_newest_commit:
        data = updater.gencfg_trunk.get(group)
        return data and data.get()
    else:
        return load_group_trunk(commit, group)


@app.route('/trunk/<int:commit>/groups/<group>')
def api_get_trunk_commit_groups(commit, group):
    if commit < updater.gencfg_trunk_oldest_commit:
        return flask.Response('commit is too old', status=410)
    data = _get_trunk_commit_group(commit, group)
    if data is not None:
        res = {key: value for key, value in data.items() if key in {'group', 'master', 'owners'}}
        res['hosts'] = data['hosts']
        return flask.jsonify(res)
    return flask.Response('group does not exist', status=404)


@app.route('/tags/<tag>/groups/<group>/instances')
def api_get_tags_groups_instances(tag, group):
    commit = _get_commit_of_tag(tag)
    if not commit:
        return flask.Response('tag does not exist', status=404)

    data = load_instances(commit, group)
    if data is not None:
        res = map(lambda record: {key: value for key, value in record.items() if key in
                  ['dc', 'domain', 'hostname', 'location', 'port', 'power']}, data)
        res = {'instances': res}
        return flask.jsonify(res)
    return flask.Response('group does not exist', status=404)


@app.route('/tags/<tag>/hosts/<host>/instances_tags')
def api_get_tags_host_instances_tags(tag, host):
    commit = _get_commit_of_tag(tag)
    if not commit:
        return flask.Response('tag does not exist', status=404)

    data = load_instances_tags(commit, host)
    if data is not None:
        return flask.jsonify({'instances_tags': data})
    return flask.Response('host does not exist', status=404)


def _get_commit_of_tag(tag):
    if tag not in updater.tags:
        return None

    return updater.tags[tag]['commit']


@app.route('/tags/<tag>/groups_cards')
def api_get_tags_groups_cards(tag):
    def generate():
        commit = updater.tags[tag]['commit']
        all_groups = get_all_topology_groups(tag_to_version(tag))

        yield '{"groups_cards": ['

        for index, group_name in enumerate(all_groups):
            card = load_card(commit, group_name, memo=False)
            if card is not None:
                yield (',' if index > 0 else '') + json.dumps(card, indent=4)

        yield ']}'

    if tag not in updater.tags:
        return flask.Response('tag does not exist', status=404)

    return flask.Response(generate(), content_type='application/json')


@app.route('/tags/<tag>/groups/<group>/card')
def api_get_tags_groups_card(tag, group):
    if tag not in updater.tags:
        return flask.Response('tag does not exist', status=404)
    commit = updater.tags[tag]['commit']
    card = load_card(commit, group)
    if card is not None:
        return flask.jsonify(card)
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/groups/<group>/instances')
def api_get_trunk_groups_instances(group):
    data = updater.gencfg_trunk.get(group)
    if data is not None:
        data = data.get()
        res = map(
            lambda inst: {
                key: value for key, value in inst.items() if key in
                ['dc', 'domain', 'hostname', 'location', 'port', 'power']
            }, data['instances']
        )
        res = {'instances': res}
        return flask.jsonify(res)
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/groups/<group>/card')
def api_get_trunk_groups_card(group):
    data = updater.gencfg_trunk.get(group)
    if data is not None:
        data = data.get_raw()
        return flask.jsonify(data['card'])
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/<int:commit>/groups/<group>/card')
def api_get_trunk_commit_groups_card(commit, group):
    if commit < updater.gencfg_trunk_oldest_commit:
        return flask.Response('commit is too old', status=410)
    data = _get_trunk_commit_group(commit, group)
    if data is not None:
        return flask.jsonify(data['card'])
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/<int:commit>/groups/<group>/instances')
def api_get_trunk_commit_groups_instances(commit, group):
    if commit < updater.gencfg_trunk_oldest_commit:
        return flask.Response('commit is too old', status=410)
    data = _get_trunk_commit_group(commit, group)
    if data is not None:
        res = map(
            lambda inst: {
                key: value for key, value in inst.items() if key in
                ['dc', 'domain', 'hostname', 'location', 'port', 'power']
            }, data['instances']
        )
        res = {'instances': res}
        return flask.jsonify(res)
    return flask.Response('group does not exist', status=404)


def get_tags():
    keys = updater.tags.keys()
    return sorted(keys, reverse=True, key=tag_to_version)


@app.route('/trunk/tags')
def api_get_trunk_tags():
    tags_ = get_tags()
    res = {'displayed_tags': tags_, 'tags': tags_}
    return flask.jsonify(res)


@app.route('/trunk/commits')
def api_get_trunk_commits():
    commits_ = updater.gencfg_trunk_commits
    res = {'commits': commits_}
    return flask.jsonify(res)


@app.route('/trunk/groups')
def api_get_trunk_groups_names():
    return flask.jsonify({
        'group_names': updater.gencfg_trunk.keys()
    })


@app.route('/trunk/<int:commit>/groups')
def api_get_trunk_commit_groups_names(commit):
    groups_ = get_all_topology_groups(commit_to_trunk(commit))
    if groups_:
        return flask.jsonify({
            'group_names': groups_
        })
    return flask.Response('commit not found', 404)


@app.route('/tags/<tag>/groups')
def api_get_tag_groups_names(tag):
    groups_ = get_all_topology_groups(tag_to_version(tag))
    if groups_:
        return flask.jsonify({
            'group_names': groups_
        })
    return flask.Response('tag not found', 404)


@app.route('/unstable/tags')
def api_get_unstable_tags():
    tags_ = get_tags()
    res = {'displayed_tags': tags_, 'tags': tags_}
    return flask.jsonify(res)


@app.route('/trunk/groups_cards')
def api_get_groups_cards():
    def generate():
        gencfg_trunk = updater.gencfg_trunk
        yield '{"groups_cards": ['
        for i, data in enumerate(gencfg_trunk.itervalues()):
            yield (',' if i > 0 else '') + json.dumps(data.get_raw()['card'], indent=4)
        yield ']}'

    return flask.Response(generate(), content_type='application/json')


def _owner_is_department(owner):
    return owner in updater.staff_group_person_map


def _guess_users(department):
    return updater.staff_group_person_map.get(department, [])


@app.route('/trunk/groups_owners')
def api_get_groups_owners():
    users = collections.defaultdict(list)
    departments = collections.defaultdict(list)
    resolved_owners = collections.defaultdict(list)

    guessed_owners = collections.defaultdict(list)
    gencfg_trunk = updater.gencfg_trunk
    for group_name, group_data in gencfg_trunk.iteritems():
        card = group_data.get_raw()['card']

        for owner in card['owners']:
            if _owner_is_department(owner):
                departments[owner].append(group_name)
                for user in _guess_users(owner):
                    guessed_owners[user].append(group_name)
            else:
                users[owner].append(group_name)

        for owner in card.get('resolved_owners', []):
            resolved_owners[owner].append(group_name)

    return flask.jsonify({
        'users': users,
        'departments': departments,
        'guessed_users': guessed_owners,
        'resolved_owners': resolved_owners,
    })


@app.route('/tags/<tag>/searcherlookup/groups/<group>/instances')
def api_get_searcherlookup_instances(tag, group):
    if tag not in updater.tags:
        return flask.Response('tag does not exist', status=404)
    commit = updater.tags[tag]['commit']
    res = load_instances(commit, group)
    if res is not None:
        return flask.jsonify({'instances': res})
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/searcherlookup/groups/<group>/instances')
def api_get_trunk_searcherlookup_groups_instances(group):
    data = updater.gencfg_trunk.get(group)
    if data is not None:
        data = data.get()
        return flask.jsonify({'instances': data['instances']})
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/<int:commit>/searcherlookup/groups/<group>/instances')
def api_get_trunk_commit_searcherlookup_groups_instances(commit, group):
    if commit < updater.gencfg_trunk_oldest_commit:
        return flask.Response('commit is too old', status=410)
    data = _get_trunk_commit_group(commit, group)
    if data is not None:
        return flask.jsonify({'instances': data['instances']})
    return flask.Response('group does not exist', status=404)


@app.route('/tags/<tag>/searcherlookup/groups/<group>/bsconfig')
def api_get_searcherlookup_bsconfig(tag, group):
    if tag not in updater.tags:
        return flask.Response('tag does not exist', status=404)
    commit = updater.tags[tag]['commit']
    instances = load_instances(commit, group)
    if instances is not None:
        return flask.Response(json_to_bsconfig(instances), content_type='text/plain')
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/searcherlookup/groups/<group>/bsconfig')
def api_get_trunk_searcherlookup_bsconfig(group):
    data = updater.gencfg_trunk.get(group)
    if data is not None:
        data = data.get()
        return flask.Response(json_to_bsconfig(data['instances']), content_type='text/plain')
    return flask.Response('group does not exist', status=404)


@app.route('/trunk/dns/ips')
def api_dns_ips():
    if updater.gencfg_dns and len(updater.gencfg_dns) > 50000:
        return flask.Response(updater.gencfg_dns, content_type='application/json')
    return flask.Response('something is wrong', status=500)


@app.route('/trunk/dns/host/<host>')
def api_dns_host(host):
    host = host.lower()

    ipaddresses = list(ips_of_hostname(host))
    if ipaddresses:
        return flask.jsonify({
            'ipAddresses': ipaddresses
        })
    else:  # safety net
        if host.startswith('bb-'):
            ipaddresses = list(ips_of_hostname(host[3:]))
            if ipaddresses:
                return flask.jsonify({
                    'ipAddresses': ipaddresses
                })

        return flask.Response('Unknown host', status=404)


@app.route('/trunk/dns/ip/<ip>')
def api_dns_ip(ip):
    ip = canonize_ipv6(ip)
    host = hostname_of_ip(ip)
    if not host:
        return flask.Response('Unknown ip', status=404)

    return flask.jsonify({
        'host': host
    })


@app.route('/trunk/dns/guess_group/<host>')
def api_dns_guess_group(host):
    host = host.lower()
    group = group_of_hostname(host)
    if not group:
        return flask.Response('Cannot guess group', status=404)

    return flask.jsonify({
        'group': group
    })


@app.route('/trunk/hosts_data', methods=['GET'])
def get_hosts_data():
    def generate():
        yield '{"hosts_data": ['
        for i, host in enumerate(hardware.hosts_hardware_fqdns().itervalues()):
            yield (',' if i > 0 else '') + json.dumps(host, indent=4)
            if i % 10001 == 0:
                gevent.sleep(0.1)
        yield ']}'

    return flask.Response(generate(), content_type='application/json')


@app.route('/trunk/hosts_data', methods=['POST'])
def post_hosts_data():
    found, not_found = [], []

    for fqdn in flask.request.json['hosts']:
        host = hardware.hosts_hardware_fqdns().get(fqdn)
        if host:
            found.append(host)
        else:
            not_found.append(fqdn)

    return flask.jsonify({
        'hosts_data': found,
        'notfound_hosts': not_found,
    })


def find_host_groups_ports(data, host):
    groups_instances = collections.defaultdict(list)
    for group in data:
        for host_, port_ in data[group]:
            if host_ == host:
                groups_instances[group].append(port_)
    return groups_instances


# @app.route('/trunk/hosts/<host>')
# def get_trunk_hosts(host):
#     return flask.jsonify(updater.trunk_host_groups_ports.get(host, {}))


# @app.route('/trunk/<int:commit>/hosts/<host>')
# def get_trunk_commit_hosts(commit, host):
#     if commit == updater.gencfg_trunk_newest_commit:
#         return get_trunk_hosts(host)
#     topology = get_instances_by_version(commit_to_trunk(commit))
#     if not topology:
#         return flask.Response('commit not found')
#     return flask.jsonify(find_host_groups_ports(topology, host))


@app.route('/tags/<tag>/hosts/<host>')
def get_tags_hosts(tag, host):
    topology = get_instances_by_version(tag_to_version(tag))
    if not topology:
        return flask.Response('tag not found')
    return flask.jsonify(find_host_groups_ports(topology, host))


@app.route('/online/groups/<group>/card')
@app.route('/online/searcherlookup/groups/<group>/card')
def get_online_group_card(group):
    card = _get_online_group_card(group)

    if card is not None:
        return flask.jsonify(card)

    return flask.Response('group does not exist', status=404)


@app.route('/online/groups/<group>/instances')
@app.route('/online/searcherlookup/groups/<group>/instances')
def get_online_group_instances(group):
    instances = _get_online_group_instances(group)

    if instances is not None:
        return flask.jsonify({'instances': instances})

    return flask.Response('group does not exist', status=404)


@app.route('/trunk/slbs')
def api_slbs():
    if updater.gencfg_slbs and len(updater.gencfg_slbs) > 2500:
        data = {'slbs': updater.gencfg_slbs}
        return flask.Response(json.dumps(data, indent=4), content_type='application/json')
    return flask.Response('something is wrong', status=500)


@app.route('/tags/tag_by_commit/<int:commit>')
def api_tag_by_commit(commit):
    try:
        tag_name = get_tag_by_commit(commit)
        response = {'tag': tag_name, 'commit': commit}
        return flask.Response(json.dumps(response, indent=4), content_type='application/json')
    except Exception:
        return flask.Response('something is wrong', status=500)


@app.route('/trunk/hbf_macroses')
def api_hbf_macroses():
    return flask.Response(
        json.dumps(dict(hbf_macroses=updater.gencfg_hbf_macroses), indent=2),
        content_type='application/json'
    )


@app.route('/trunk/hosts/hosts_to_groups', methods=['GET', 'POST'])
@app.route('/trunk/hosts/hosts_to_groups/<hostname_list>')
def api_hosts_hosts_to_groups_list(hostname_list=None):
    if not updater.gencfg_hosts_to_groups:
        return flask.Response('something is wrong', status=500)

    if hostname_list is None and not flask.request.data:
        data = {
            'groups_by_host': updater.gencfg_hosts_to_groups,
            'hosts_by_group': {},
            'unknown_hosts': []
        }

    else:
        hostname_list = hostname_list.split(',') if hostname_list else \
            json.loads(flask.request.data).get('hosts', [])

        data = {'groups_by_host': {}, 'hosts_by_group': {}, 'unknown_hosts': []}
        for hostname in hostname_list:
            if hostname in updater.gencfg_hosts_to_groups:
                data['groups_by_host'][hostname] = updater.gencfg_hosts_to_groups[hostname]
            else:
                data['unknown_hosts'].append(hostname)

    for hostname, group_list in data['groups_by_host'].items():
        for group_name in group_list:
            if group_name not in data['hosts_by_group']:
                data['hosts_by_group'][group_name] = []
            data['hosts_by_group'][group_name].append(hostname)
    data['hosts'] = list(data['groups_by_host'].keys())
    data['groups'] = list(data['hosts_by_group'].keys())

    return flask.Response(json.dumps(data, indent=4), content_type='application/json')


@app.route('/trunk/hosts/hosts_to_hardware', methods=['GET', 'POST'])
@app.route('/trunk/hosts/hosts_to_hardware/<hostname_list>')
def api_hosts_hosts_to_hardware_list(hostname_list=None):
    if not updater.gencfg_hosts_to_hardware:
        return flask.Response('something is wrong', status=500)

    if hostname_list is None and not flask.request.data:
        data = {
            'hardware_by_host': updater.gencfg_hosts_to_hardware,
            'unknown_hosts': []
        }
        return flask.Response(json.dumps(data, indent=4), content_type='application/json')

    hostname_list = hostname_list.split(',') if hostname_list else \
        json.loads(flask.request.data).get('hosts', [])

    data = {'hardware_by_host': {}, 'unknown_hosts': []}
    for hostname in hostname_list:
        if hostname in updater.gencfg_hosts_to_hardware:
            data['hardware_by_host'][hostname] = updater.gencfg_hosts_to_hardware[hostname]
        else:
            data['unknown_hosts'].append(hostname)
    data['hosts'] = list(data['hardware_by_host'].keys())

    return flask.Response(json.dumps(data, indent=4), content_type='application/json')


def _get_online_group_card(group):
    version = _get_online_group_version(group)
    if version is None:
        return None
    if is_trunk(version):
        return _get_trunk_commit_group(trunk_to_commit(version), group)['card']

    commit = _get_commit_of_tag(version_to_tag(version))
    if not commit:
        return None

    return load_card(commit, group)


def _get_online_group_instances(group):
    version = _get_online_group_version(group)
    if version is None:
        return None
    if is_trunk(version):
        return _get_trunk_commit_group(trunk_to_commit(version), group)['instances']

    commit = _get_commit_of_tag(version_to_tag(version))
    if not commit:
        return None

    return load_instances(commit, group)


def _get_online_group_version(group):
    versions = {}
    for version, group_state in updater.groups.group_versions(group).iteritems():
        total = sum(1 for _ in group_state.iter_all_instances())
        alive = sum(1 for _ in group_state.iter_alive_instances())
        if alive:
            versions[version] = {'total': total, 'alive': alive}

    return max(versions.items(), key=_group_sort_key)[0] if versions else None


def _group_sort_key(x):
    version, data = x
    total, alive = data['total'], data['alive']

    return (1 if alive > 0.88 * total else 0), version


@app.route('/_solomon')
def _last_full_update():
    return flask.jsonify({'sensors': [
        {'labels': {'sensor': 'since_update'}, 'value': updater.since_last_update.total_seconds()},
        {'labels': {'sensor': 'dns_age'}, 'value': updater.dns_age.total_seconds()},
        {'labels': {'sensor': 'snapshot_age'}, 'value': updater.snapshot_age.total_seconds()},
        {'labels': {'sensor': 'alive_instances'}, 'value': 1},
    ]})


@app.route('/cpumodels')
def api_cpumodels():
    return flask.jsonify(get_cpumodels())
