import flask
from web import app

from libraries.topology.groups import get_commit_of_version, tag_to_version
from libraries.topology.searcher_lookup import load_instances
from libraries.json_diff import diff
from libraries import resolve_yp

from updaters import updater


def group_mtn_addresses(group, version):
    instances = load_instances(get_commit_of_version(version), group)
    if not instances:
        return None
    return {
        '{}:{}'.format(instance['hostname'], instance['port']): instance['hbf']
        for instance in instances
    }


@app.route('/group/<group>/alive')
def api_get_alive_group_versions(group):
    return flask.jsonify({'versions': resolve_yp.group_alive_versions(group)})


def _get_moved_to(group):
    trunk_group = updater.gencfg_trunk.get(group)
    if trunk_group and trunk_group.get()['card']['yp']['moved']:
        where = trunk_group.get()['card']['yp']['where']
        return where
    return None


@app.route('/group/<group>/alive/mtn')
def api_get_group_alive_mtn(group):
    where = _get_moved_to(group)
    try:
        if where:
            records = resolve_yp.resolve_moved_to_yp(group, where)
        else:
            records = resolve_yp.resolve_gencfg_group(group, mtn=True)
    except resolve_yp.ResolveError as exc:
        return flask.Response(exc.message, status=404)

    result = {
        '{}:{}'.format(item.record.fqdn, item.record.port): {
            'interfaces': {
                'backbone': {
                    'hostname': item.record.fqdn,
                    'ipv6addr': item.record.ipv6,
                }
            }
        }
        for item in records
    }
    return flask.jsonify(result)


@app.route('/group/<group>/<tag>/mtn')
def api_get_group_tag_mtn(group, tag):
    instances = group_mtn_addresses(group, tag_to_version(tag))
    if instances is not None:
        return flask.jsonify(instances)
    return flask.Response('group not found', 404)


@app.route('/group/<group>/diff/<a>/<b>')
def api_get_diff_group(group, a, b):
    a = load_instances(get_commit_of_version(tag_to_version(a)), group)
    b = load_instances(get_commit_of_version(tag_to_version(b)), group)
    if a:
        a = {'{}:{}'.format(inst['hostname'], inst['port']): inst for inst in a}
    if b:
        b = {'{}:{}'.format(inst['hostname'], inst['port']): inst for inst in b}
    if not a and not b:
        return flask.jsonify({})
    if not a:
        return flask.jsonify(b)
    if not b:
        return flask.jsonify({'deleted': 1})
    return flask.jsonify({'diff': diff(a, b)})
