from gevent import monkey
monkey.patch_all(dns=False)

import os
import time
import logging
import logging.handlers
import argparse

import gevent
from gevent.pywsgi import WSGIServer
from web import app
from flask import request

from updaters import updater


NAME_OF_SERVICE = 'dispenser'


def ensure_dir(path):
    if not os.path.exists(path):
        os.makedirs(path)


def params():
    parser = argparse.ArgumentParser(description=NAME_OF_SERVICE)
    parser.add_argument('-p', '--port', default=7475, type=int, help='port to listen')
    parser.add_argument('--warmup', action='store_true', help='warmup before accepting connections')
    parser.add_argument('--no-warmup', dest='warmup', action='store_false', help="don't warm up")
    parser.add_argument('--debug', dest='debug', default=False, action='store_true', help='enable fancy logging to stdout')
    parser.add_argument('--beta', dest='beta', default=False, action='store_true', help='reload templates')
    parser.set_defaults(warmup=True)
    return parser.parse_args()


def setup_logging(debug):
    ensure_dir('./logs')

    formatter = SkynetishFormatter()

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    debug_handler = logging.handlers.RotatingFileHandler('./logs/debug.log', maxBytes=1024**3, backupCount=10)
    debug_handler.setLevel(logging.DEBUG)
    debug_handler.setFormatter(formatter)
    logger.addHandler(debug_handler)

    info_handler = logging.handlers.RotatingFileHandler('./logs/instance.log', maxBytes=1024**3, backupCount=10)
    info_handler.setLevel(logging.INFO)
    info_handler.setFormatter(formatter)
    logger.addHandler(info_handler)

    if debug:
        console_handler = logging.StreamHandler()
        console_handler.setLevel(logging.DEBUG)
        console_handler.setFormatter(formatter)
        logger.addHandler(console_handler)

    logging.getLogger('requests').setLevel(logging.WARNING)
    logging.getLogger('werkzeug').addHandler(debug_handler)

    @app.after_request
    def after_request_logging(response):
        if response.status_code != 200:
            app.logger.info(' '.join([
                request.remote_addr,
                request.method,
                request.url,
                str(response.status_code)
                ])
            )
        return response


class SkynetishFormatter(logging.Formatter):
    def __init__(self):
        super(SkynetishFormatter, self).__init__()

    def formatTime(self, record, **kwargs):
        t = time.strftime('%Y-%m-%d %H:%M:%S', self.converter(record.created))
        return '%s.%03d' % (t, record.msecs)

    def format(self, record):
        levelno = record.levelno
        if levelno > 5:
            level = '[%-4s]' % logging.getLevelName(levelno)
        else:
            level = '(%s)' % (str(levelno) if levelno < 0 else ' %d' % levelno)

        date = self.formatTime(record)
        message = record.getMessage()
        if record.exc_info:
            message += '\n' + self.formatException(record.exc_info)
        header = '{0} {1} [{2}]  '.format(date, level, record.name)

        if '\n' in message:
            # special case for multi-line log messages
            message_lines = message.splitlines()
            line = [header + message_lines[0]]
            prepend = '%s%s' % (' ' * (len(header) - 2), ': ')
            line.extend(['%s%s' % (prepend, l) for l in message_lines[1:]])
            line = '\n'.join(line)
        else:
            line = '{header}{message}'.format(header=header, message=message)
        return line


def warmup_hardware():
    import libraries.hardware
    libraries.hardware.load_singletons()


def main():
    pars = params()

    setup_logging(pars.debug)
    warmup_hardware()

    http = WSGIServer(('::', pars.port), app)
    gevent.spawn(updater().start)
    if pars.warmup:
        while not updater().ready:
            gevent.sleep(5)
    http.serve_forever()


if __name__ == '__main__':
    main()
