import flask
import datetime
from libraries.containers import containers, clusters

from util import format_active_app, format_active_vm, split_hosts_to_states
from updaters import updater


containers_bp = flask.Blueprint(__name__, 'containers')


@containers_bp.route('/')
def api_containers_handler():
    lst = containers.Container.list(json=True)
    return flask.jsonify({'containers': lst})


@containers_bp.route('/<type_>/clusters')
def list_type_clusters(type_):
    res = {
        rec['name']: rec
        for rec in clusters.Cluster.list_json({'type': type_})
    }
    return flask.jsonify(res)


@containers_bp.route('/<cluster>')
def api_containers_cluster_handler(cluster):
    if cluster not in updater().clusters:
        return flask.Response(status=404)
    lst = containers.Container.list({
        'reported_at': {'$gte': datetime.datetime.now() - datetime.timedelta(minutes=2)},
        'cluster': cluster
    }, json=True)
    return flask.jsonify({'containers': lst})


@containers_bp.route('/<cluster>/active')
def api_containers_cluster_active_handler(cluster):
    if cluster not in updater().clusters:
        return flask.Response(status=404)
    cluster = updater().clusters[cluster]
    lst = containers.Container.list({
        'active': True,
        'reported_at': {'$gte': datetime.datetime.now() - datetime.timedelta(minutes=2)},
        'cluster': cluster.name
    }, json=True)
    if cluster.mode == 'app':
        lst = format_active_app(lst, cluster.name)
    else:
        lst = format_active_vm(lst, cluster.name)
    return flask.jsonify({'containers': lst})


@containers_bp.route('/<cluster>/status/hosts')
def api_containers_cluster_status_hosts(cluster):
    if cluster not in updater().clusters:
        return flask.Response(status=404)
    group = flask.request.args.get('group')
    group = [group] if group else None
    res = split_hosts_to_states(cluster, group)
    return flask.jsonify({
        'hosts': {key: [a['name'] for a in res[key].values()] for key in res}
    })


@containers_bp.route('/<cluster>/status')
def api_containers_cluster_status(cluster):
    if cluster not in updater().clusters:
        return flask.Response(status=404)
    group = flask.request.args.get('group')
    group = [group] if group else None
    res = split_hosts_to_states(cluster, group)
    return flask.jsonify({
        'status': {key: len(res[key]) for key in res},
    })


@containers_bp.route('/<cluster>/status/resources')
def api_containers_cluster_status_resources(cluster):
    group = flask.request.args.get('group')
    group = [group] if group else None
    hosts = split_hosts_to_states(cluster, group)
    hosts = hosts['active'].values() + hosts['slowed'].values()
    total = 0
    res = {'cpu': {
        'total': {
            'usage': 0, 'limit': 0
        },
        'avg': {
            'usage': 0, 'limit': 0
        }}}
    for one in hosts:
        if one['usage']['cpu'] > 20:
            res['cpu']['total']['limit'] += one['limits']['cpu']['max'] or 0
            res['cpu']['total']['usage'] += one['usage']['cpu'] or 0
            total += 1
    res['cpu']['avg']['limit'] = res['cpu']['total']['limit'] / total
    res['cpu']['avg']['usage'] = res['cpu']['total']['usage'] / total
    res['cpu']['ratio'] = res['cpu']['avg']['usage'] / res['cpu']['avg']['limit']
    return flask.jsonify({
        'resources': res,
    })
