import datetime
import logging
from collections import defaultdict
from libraries.containers.containers import Container
from libraries.hardware import cpumodel_of_host
from updaters import updater


def format_cpu(dct, host):
    ncores = cpumodel_of_host(host).ncpu
    power = cpumodel_of_host(host).power
    return {'min': dct['min'], 'max': dct['max'],
            'min_cores': dct['min'] / 100.0,
            'max_cores': dct['max'] / 100.0,
            'min_power': dct['min'] * power / ncores / 100,
            'max_power': dct['max'] * power / ncores / 100}


def format_mem(dct):
    return {'min': dct['min'], 'max': dct['max']}


def instances_for_cluster(cluster, groups=None):
    res = {}
    for group, conf in cluster.groups.items():
        if not groups or group in groups:
            data = updater().instances_for_group(group, updater().find_version_for_group(conf.agents_group))
            for host, instance in data.iteritems():
                res[host] = instance
    return res


def _format_active(container, instance, host):
    return {
        'limits': {
            'cpu': format_cpu(container['limits']['cpu'], host),
            'mem': format_mem(container['limits']['mem']),
            'net': container['limits'].get('net')
        },
        'host_info': {
            'location': {
                'dc': instance['dc'],
                'rack': instance['rack'],
                'line': instance['queue'],
                'domain': instance['domain'],
            },
            'resources': instance['host_resources'],
        }
    }


def format_active_vm(lst, cluster, groups=None):
    res = []
    instances = instances_for_cluster(updater().clusters[cluster], groups)
    for one in lst:
        if one['mode'] != 'os' or not one['limits']:
            continue
        host = one['host']
        if host in instances:
            doc = _format_active(one, instances[host], host)
            doc.update(host=one.get('dns_name'), mode='os')
            doc['host_info'].update(vlans=one.get('vlans'))
            res.append(doc)
    return res


def format_active_app(lst, cluster, groups=None):
    res = []
    instances = instances_for_cluster(updater().clusters[cluster], groups)
    for one in lst:
        if one['mode'] != 'app' or not one['limits']:
            continue
        host = one['host']
        name = 'psi-{}'.format(host)
        if host in instances:
            doc = _format_active(one, instances[host], host)
            doc.update(host=name, mode='app')
            res.append(doc)
    return res


def split_hosts_to_states(cluster, groups=None):
    lst = Container.list({
        'cluster': cluster
    }, json=True)
    instances = instances_for_cluster(updater().clusters[cluster], groups)
    res = defaultdict(dict, stopped={}, dead={}, paused={},
                      unknown={}, slowed={}, not_listed={},
                      network_broken={}, skynet_broken={},
                      volume_broken={}, none={}, dns_problem={})

    now = datetime.datetime.now()
    res['offline'] = {a['name']: a for a in lst if a['reported_at'] < now - datetime.timedelta(minutes=2)}
    lst = [a for a in lst if a['reported_at'] >= now - datetime.timedelta(minutes=2)]
    res['not_in_group'] = {a['name']: a for a in lst if a['host'] not in instances}
    lst = [a for a in lst if a['host'] in instances]
    res['active'] = {a['name']: a for a in lst if a['active']}
    lst = [a for a in lst if not a['active']]
    for one in lst:
        res[one['state']][one['name']] = one
    return res


_log = logging.getLogger(__name__)
