import flask
import json
import re
from web.limits.util import (
    get_limits, calc_available_resources, get_resources, MEM_RESERVE, CPU_RESERVE,
)
from libraries.containers.clusters import Allocation, ClusterType
from libraries.containers.containers import Container
from updaters import updater


control_bp = flask.Blueprint(__name__, 'control_bp')


@control_bp.route('/on_demand/clusters')
def _list_clusters():
    return flask.jsonify({'clusters': list_clusters()})


def list_clusters():
    clusters = []
    for cluster in updater().clusters.values():
        if cluster.type == ClusterType.on_demand:
            clusters.append(cluster.name)
    return clusters


@control_bp.route('/on_demand/clusters/<cluster>')
def _get_cluster(cluster):
    return flask.jsonify(get_cluster(cluster))


def get_cluster(cluster):
    cluster = updater().clusters[cluster]
    result = {}
    for group in cluster.groups:
        hosts = {host for host in updater().instances_for_group(group, updater().find_version_for_group(group))}
        for host in hosts:
            result[host] = calc_summary(host)
    return result


def _get_cluster_group(host):
    groups_on_host = updater().type_host_groups[ClusterType.on_demand][host]
    assert len(groups_on_host) == 1
    group = list(groups_on_host)[0]
    for cluster in updater().clusters.values():
        if group in cluster.groups:
            return cluster, group
    return None


def calc_summary(host):
    cluster, group = _get_cluster_group(host)
    res, _ = get_limits(host, calc_info=True)
    res = calc_available_resources(res, get_resources(host), host)
    allocations = Allocation.list({'host': host})
    res['slots'] = {}
    res['slots']['allocated'] = [extend_vm(alloc) for alloc in sorted(allocations, key=lambda x: x.name)]
    all_ports = cluster.groups[group].ports or updater().group_slots[group][host].keys()[:cluster.groups[group].max_containers_cnt]
    all_ports = sorted(all_ports)
    res['slots']['left'] = list(set(all_ports).difference({
        int(alloc.port) for alloc in allocations
    }))
    res['cpu']['left'] = res['cpu']['free'] - CPU_RESERVE
    res['mem']['left'] = res['mem']['free'] - MEM_RESERVE
    for alloc in allocations:
        res['mem']['left'] -= alloc.mem
        res['cpu']['left'] -= alloc.cpu
    res['host'] = host
    res['cluster'] = cluster.name
    return res


@control_bp.route('/on_demand/allocate', methods=['POST'])
def _allocate_container():
    data = json.loads(flask.request.data)
    return flask.jsonify({'success': allocate_container(data)})


def allocate_container(data):
    try:
        alloc = Allocation(**data)
    except KeyError:
        return flask.Response('Bad input', 400)
    host = data['host']
    resources = calc_summary(host)
    if resources['mem']['left'] < alloc.mem or resources['cpu']['left'] < alloc.cpu or not resources['slots']['left']:
        return flask.Response('Not enough resources', 409)
    if data['port'] not in resources['slots']['left']:
        return flask.Response('Port is not available', 400)
    if alloc.mem < 1024 ** 3 or alloc.cpu < 50:
        return flask.Response('Too low requirements', 400)
    if not re.match('[a-zA-Z0-9_-]+', alloc.name):
        return flask.Response('Bad vm name', 400)
    alloc.save()
    return True


@control_bp.route('/on_demand/vm/<name>/prolong', methods=['POST'])
def _prolong_vm(name):
    ttl = json.loads(flask.request.data)['ttl']
    return flask.jsonify({'success': prolong_vm(name, ttl)})


def prolong_vm(name, ttl):
    alloc = Allocation.list({'name': name})[0]
    alloc.prolong(ttl)
    alloc.save()
    return True


@control_bp.route('/on_demand/vm/<name>', methods=['GET'])
def _get_vm(name):
    return flask.jsonify(get_vm(name))


def extend_vm(alloc):
    res = alloc.json()
    try:
        container = Container.list({'name': res['name']})[0]
        res['dns_name'] = container.dns_name
        res['online'] = container.online
        res['active'] = container.online and container.active
        res['state'] = container.state if container.online else 'none'
    except IndexError:
        res['online'] = False
        res['status'] = 'not listed'
        res['active'] = False
    return res


def get_vm(name):
    alloc = Allocation.list({'name': name})[0]
    return extend_vm(alloc)


def get_host(host):
    return calc_summary(host)
