import os

import flask

from libraries.containers.clusters import Cluster, ClusterType

PREFIX_URL = '/on_demand'
CLUSTERS_URL = PREFIX_URL + '/clusters'
HOSTS_URL = PREFIX_URL + '/hosts'
VMS_URL = PREFIX_URL + '/vms'


def cluster_url(cluster_name):
    return os.path.join(CLUSTERS_URL, cluster_name)


def host_url(host):
    return os.path.join(HOSTS_URL, host)


def vm_url(vm_name):
    return os.path.join(VMS_URL, vm_name)


def register_handlers(bp):
    from web.control import get_cluster, get_vm, get_host

    @bp.route(CLUSTERS_URL)
    def gui_clusters():
        clusters = [
            cl.name for cl in Cluster.list()
            if cl.type == ClusterType.on_demand
            and cl.name not in ('docs_writers',)
        ]
        return flask.render_template('cluster-list.html', clusters=clusters)

    @bp.route(CLUSTERS_URL + '/<cluster>')
    def gui_on_demand_cluster(cluster):
        cluster = get_cluster(cluster)
        allocations = sum([host['slots']['allocated'] for host in cluster.values()], [])
        vms = [
            {'name': vm['name'], 'online': vm['online'], 'active': vm['active']}
            for vm in allocations
        ]
        vms.sort(key=lambda x: x['name'])
        return flask.render_template('on-demand-cluster.html', vms=vms, hosts=cluster)

    @bp.route('/on_demand/vms/<vm>')
    def gui_on_demand_vm(vm):
        vm = get_vm(vm)
        return flask.render_template('vm.html', **vm)

    @bp.route('/on_demand/hosts/<host>')
    def gui_on_demand_host(host):
        host = get_host(host)
        return flask.render_template('host.html', **host)
