import flask
import datetime
import colorsys
import json

from libraries.containers.containers import Container


GREEN = 0.0
YELLOW = 50.0
RED = 900.0

INACTIVE_COLOR = '#{:02x}{:02x}{:02x}'.format(*[10, 10, 30])


def green_to_yellow(h):
    h = YELLOW - h
    h /= YELLOW * 6
    h += 1.0 / 6
    return h


def yellow_to_red(h):
    h = RED - h
    h /= RED * 6
    return h


def get_color(h):
    s, v = 0.9, 0.9
    if h == 0:
        s, v = 0.2, 0.4
    if GREEN <= h < YELLOW:
        h = green_to_yellow(h)
    elif YELLOW <= h < RED:
        h = yellow_to_red(h)
    else:
        h = 0

    rgb = [int(color * 255) for color in colorsys.hsv_to_rgb(h, s, v)]
    return '#{:02x}{:02x}{:02x}'.format(*rgb)


def list_containers(cluster):
    now = datetime.datetime.now() - datetime.timedelta(minutes=2)
    containers = []
    for one in Container.list({'cluster': cluster}, json=True):
        cont = {'name': one['name']}
        if one['active'] and one['reported_at'] > now and one['usage']:
            cont['color'] = get_color(one['usage']['cpu'])
            cont['cpu'] = one['usage']['cpu'] or 0.0
        else:
            cont['color'] = INACTIVE_COLOR
            cont['cpu'] = 0
        containers.append(cont)
    return containers


def register_handlers(bp):
    @bp.route('/cpu_map/<cluster>')
    def gui_cpu_map(cluster):
        containers = list_containers(cluster)
        containers.sort(key=lambda x: x['cpu'])
        return flask.render_template('heatmap.html', containers=containers, static=False, cluster=cluster)

    @bp.route('/cpu_static_map/<cluster>')
    def gui_cpu_static_map(cluster):
        containers = list_containers(cluster)
        containers.sort(key=lambda x: x['name'])
        return flask.render_template('heatmap.html', containers=containers, static=True, cluster=cluster)

    @bp.route('/api_cpu_map/<cluster>')
    def api_cpu_map_cluster(cluster):
        static = flask.request.args.get('static', '0') == '1'
        containers = list_containers(cluster)
        if static:
            containers.sort(key=lambda x: x['name'])
        else:
            containers.sort(key=lambda x: x['cpu'])
        return flask.Response(json.dumps({'containers': containers}), mimetype='application/json')
