import json
import flask
import datetime
from util import (
    get_limits, get_resources, calc_available_resources, host_port_to_dns_name, calc_clusters
)
from libraries.utils import shortname
from libraries import hardware
from libraries.containers.clusters import ClusterType

from updaters import updater


configs_bp = flask.Blueprint(__name__, 'configs_bp')


@configs_bp.route('/configs/<host>')
def api_configs_handler(host):
    compact = flask.request.args.get('compact', '0') == '1'
    gencfg_ip = flask.request.args.get('gencfg_ip', '0') == '1'
    specific_cluster = flask.request.args.get('cluster', None)

    type_ = flask.request.args.get('type')
    if type_ == ClusterType.on_demand:
        res = configs_on_demand(host, gencfg_ip)
    elif type_ == ClusterType.dynamic:
        res = configs_dynamic(host, gencfg_ip)
    elif type_ == ClusterType.static:
        res = configs_static(host, gencfg_ip, specific_cluster)
    elif type_ == ClusterType.capi:
        res = configs_capi(host)
    elif type_ == ClusterType.samogon:
        res = configs_samogon(host)
    else:
        return flask.Response('type not specified', status=400)

    if compact:
        return flask.Response(json.dumps(res), mimetype='application/json')
    return flask.jsonify(res)


def configs_on_demand(host, use_gencfg_ip):
    res = {}
    allocations = updater().host_allocations.get(host, [])
    groups_on_host = updater().type_host_groups[ClusterType.on_demand].get(host, [])
    for alloc in allocations:
        if not alloc.expired:
            cluster = updater().clusters[alloc.cluster]
            for group in set(cluster.groups) & set(groups_on_host):
                if use_gencfg_ip:
                    bb_ip = updater().group_slots[group][host][alloc.port]['hbf']['interfaces']['backbone']['ipv6addr']
                    fb_ip = updater().group_slots[group][host][alloc.port]['hbf']['interfaces']['fastbone']['ipv6addr']
                    dns_name = updater().group_slots[group][host][alloc.port]['hbf']['interfaces']['backbone']['hostname']
                else:
                    bb_ip = None
                    fb_ip = None
                    dns_name = host_port_to_dns_name(host, int(float(alloc.port)))
                res[alloc.name] = vm_config(
                    alloc.mem, alloc.cpu, alloc.name, dns_name,
                    cluster.groups[group].project_id, cluster.name,
                    bb_ip, fb_ip
                )
                res[alloc.name]['limits']['cpu']['max'] = host_cpu_resources(host)['total']
                break
    return res


def is_satisfy(host, smooth):
    try:
        number = int(host.split('-')[1].split('.')[0])
    except (TypeError, IndexError, ValueError):
        return False
    prob = ((datetime.datetime.now() - smooth.since).total_seconds() / smooth.duration)
    if prob * 10000 > number:
        return True
    return False


def get_clusters(host, clusters):
    def _f(smooth_):
        return is_satisfy(host, smooth_)

    result = {}
    for name in clusters:
        if name in updater().smooth_clusters:
            lst = filter(_f, updater().smooth_clusters[name])
            if lst:
                result[name] = lst[0]
            else:
                result[name] = clusters[name]
        else:
            result[name] = clusters[name]
    return result


def _place_all_clusters(host, use_gencfg_ip):
    limits, _ = get_limits(host)
    resources = calc_available_resources(limits, get_resources(host), host)

    return calc_clusters(
        resources=resources,
        host=host,
        use_gencfg_ip=use_gencfg_ip,
        dynamic_clusters=get_clusters(host, updater().type_clusters[ClusterType.dynamic]),
        dynamic_groups=updater().type_host_groups[ClusterType.dynamic].get(host, {}),
        static_clusters=get_clusters(host, updater().type_clusters[ClusterType.static]),
        static_groups=updater().type_host_groups[ClusterType.static].get(host, {}),
        samogon_clusters=get_clusters(host, updater().type_clusters[ClusterType.samogon]),
        samogon_groups=updater().type_host_groups[ClusterType.samogon].get(host, {}),
    )


def _filter_types(values, type_):
    return {
        port: container
        for port, container in values.items()
        if container['type'] == type_
    }


def configs_dynamic(host, use_gencfg_ip):
    values = _filter_types(_place_all_clusters(host, use_gencfg_ip), 'dynamic')
    result = {}

    for port, value in values.items():
        if value['mode'] == 'app':
            name = value['cluster']
            result[name] = app_config(value['mem'], value['cpu'], name, value['cluster'])
            continue
        if use_gencfg_ip:
            name = dns_name = value['hostname']
            result[name] = vm_config(
                value['mem'], value['cpu'], name, dns_name, value['project_id'], value['cluster'],
                value['bb_ip'], value['fb_ip']
            )
        else:
            name = dns_name = host_port_to_dns_name(host, int(float(port)))
            result[name] = vm_config(
                value['mem'], value['cpu'], name, dns_name, value['project_id'], value['cluster'],
                None, None
            )
        result[name]['limits']['cpu']['min'] = 50
    return result


def configs_static(host, use_gencfg_ip, cluster_name=None):
    values = _filter_types(_place_all_clusters(host, use_gencfg_ip), 'static')
    result = {}

    for port, value in values.items():
        if cluster_name and cluster_name == value['cluster']:
            return {
                'limits': {
                    'mem': value['mem'],
                }
            }
        if use_gencfg_ip:
            name = dns_name = value['hostname']
            result[name] = vm_config(
                value['mem'], value['cpu'], name, dns_name, value['project_id'], value['cluster'],
                value['bb_ip'], value['fb_ip']
            )
        else:
            name = dns_name = host_port_to_dns_name(host, int(float(port)))
            result[name] = vm_config(
                value['mem'], value['cpu'], name, dns_name, value['project_id'], value['cluster'],
                None, None
            )
        result[name]['limits']['cpu']['min'] = 50

    return result


def configs_capi(host):
    limits, _ = get_limits(host)
    resources = calc_available_resources(limits, get_resources(host), host)

    values = calc_clusters(
        resources=resources,
        dynamic_clusters=get_clusters(host, updater().type_clusters[ClusterType.capi]),
        dynamic_groups=updater().type_host_groups[ClusterType.capi].get(host, {}),
        host=host,
        use_gencfg_ip=False,
        static_clusters=get_clusters(host, updater().type_clusters[ClusterType.static]),
        static_groups=updater().type_host_groups[ClusterType.static].get(host, {}),
        samogon_clusters=get_clusters(host, updater().type_clusters[ClusterType.samogon]),
        samogon_groups=updater().type_host_groups[ClusterType.samogon].get(host, {}),
    )
    values = _filter_types(values, 'capi')
    result = {}
    assert len(values) in (0, 1)
    for port, value in values.items():
        name = 'iss-agent'
        result[name] = app_config(value['mem'], value['cpu'], name, value['cluster'])
    return result


def configs_samogon(host):
    values = _filter_types(_place_all_clusters(host, use_gencfg_ip=True), 'samogon')
    result = {}

    for port, value in values.items():
        name = dns_name = value['hostname']
        result[name] = vm_config(
            value['mem'], value['cpu'], name, dns_name, value['project_id'], value['cluster'],
            value['bb_ip'], value['fb_ip']
        )
        result[name]['limits']['cpu']['min'] = 50
    return result


def _config(mem, cpu, name, cluster_name):
    if not isinstance(mem, dict):
        mem = {'min': mem, 'max': mem}
    if not isinstance(cpu, dict):
        cpu = {'min': cpu, 'max': cpu}
    return {
        'limits': {
            'cpu': cpu,
            'mem': mem
        },
        'conf': {
            'name': name,
            'cluster': cluster_name
        }
    }


def vm_config(mem, cpu, name, dns_name, project_id, cluster_name, bb_ip, fb_ip):
    conf = _config(mem, cpu, name, cluster_name)
    conf['conf'].update(mode='os', dns_name=dns_name, project_id=project_id)
    if bb_ip or fb_ip:
        conf['conf'].update(bb_ip=bb_ip, fb_ip=fb_ip)
    return conf


def app_config(mem, cpu, name, cluster_name):
    conf = _config(mem, cpu, name, cluster_name)
    conf['conf']['mode'] = 'app'
    return conf


def host_cpu_resources(host):
    cpu = {}
    resources = hardware.hosts_hardware()[shortname(host)]
    cpu['total'] = hardware.models()[resources['model']].ncpu * 100
    cpu['power'] = hardware.models()[resources['model']].power
    return cpu


@configs_bp.route('/resources/<host>')
def api_resources_handler(host):
    cpu = host_cpu_resources(host)
    return flask.jsonify({'cpu': cpu})
