import datetime

from updaters import updater
from libraries import hardware
from libraries.utils import shortname
from libraries.topology.utils import tag_to_version
from libraries.topology.groups import get_commit_of_version
from libraries.containers.limits import limits, intersect_groups


MEM_RESERVE = 5 * 1024 ** 3
MEM_RESERVE_FOR_TABLES_RATIO = 0.02
MEM_AGENT_RESERVE = 0.2 * 1024 ** 3
CPU_RESERVE = 200
MEM_RESERVE_PER_VM = 1 * 1024 ** 3
CPU_RESERVE_PER_VM = 50

ROTOR_CLUSTERS = ['rotor-test', 'rotor']
MEM_YT_NEEDS = 60 * (1024 ** 3)
MEM_ROTOR_NEEDS = 6 * (1024 ** 3)


def is_alive(state):
    alive_cnt = state.alive_instances_cnt()
    all_cnt = state.all_instances_cnt()

    if alive_cnt == 0:
        return False

    if float(alive_cnt) / all_cnt > 0.05:
        return True

    if alive_cnt > 0 and all_cnt < 10:
        return True

    return False


def actual_groups(host):
    min_version = tag_to_version(updater().topology.min_tag)
    groups = updater().online_state.groups.groups_on_host(host)
    used = {}
    for (group, version), state in groups.items():
        if not is_alive(state):
            continue
        if get_commit_of_version(used.get(group, min_version)) < get_commit_of_version(version):
            used[group] = version
    groups = used.items()
    return groups


def future_groups(host):
    groups = updater().topology.trunk_host_group_mapping.get(host, [])
    return groups


def get_limits(host, calc_info=False):
    return limits(
        host, intersect_groups(actual_groups(host), future_groups(host)),
        updater().topology.trunk_topology, calc_info=calc_info
    )


def get_resources(host):
    return hardware.hosts_hardware_fqdns().get(host) or hardware.hosts_hardware()[shortname(host)]


def calc_available_resources(res, resources, host):
    res['mem']['total'] = resources['memory'] * 1024 ** 3
    res['cpu']['total'] = hardware.models()[resources['model']].ncpu * 100

    mem_free = res['mem']['total'] * (1 - MEM_RESERVE_FOR_TABLES_RATIO) - MEM_RESERVE
    res['mem']['free'] = mem_free - res['mem']['guarantee']

    res['cpu']['free'] = res['cpu']['total'] - res['cpu']['guarantee'] - CPU_RESERVE
    return res


def place_cluster(cluster, group, host, mem_left, gencfg_ip):
    result = {}
    if not cluster.groups[group].active:
        return result, mem_left

    ports = sorted(cluster.groups[group].ports or updater().group_slots[group][host])
    if cluster.groups[group].max_containers_cnt is not None:
        ports = ports[:cluster.groups[group].max_containers_cnt]
    for port in ports:
        if mem_left < cluster.mem_min:
            return result, mem_left
        vm_mem = min(int(mem_left), cluster.mem_max)
        result[port] = {
            'cpu': {'max': int(cluster.cpu_max), 'min': int(cluster.cpu_min)},
            'mem': {'max': int(vm_mem), 'min': int(vm_mem)},
            'cluster': cluster.name,
            'project_id': cluster.groups[group].project_id,
            'mode': cluster.mode,
            'type': cluster.type,
        }
        if gencfg_ip:
            result[port].update(
                bb_ip=updater().group_slots[group][host][port]['hbf']['interfaces']['backbone']['ipv6addr'],
                fb_ip=updater().group_slots[group][host][port]['hbf']['interfaces']['fastbone']['ipv6addr'],
                hostname=updater().group_slots[group][host][port]['hbf']['interfaces']['backbone']['hostname']
            )
        mem_left -= vm_mem + MEM_RESERVE_PER_VM
    return result, mem_left


def filter_clusters(clusters, names):
    return {
        name: cluster for name, cluster in clusters.items()
        if name in names
    }


def sorted_clusters(clusters):
    order = [
        'rotor-test', 'yt-socrates', 'yt-xcalc', 'xcalc', 'rotor', 'matrixnet', 'capi_test', 'test',
    ]
    for name in order:
        if name in clusters:
            yield clusters[name]
    for name in sorted(set(clusters.keys()) - set(order)):
        yield clusters[name]


def _get_yt_on_host(static_clusters, static_groups):
    for group in static_groups:
        for cluster_name in static_clusters:
            yt_cluster = updater().clusters[cluster_name]
            if group in yt_cluster.groups:
                return yt_cluster, group
    return None, None


def _get_samogon_on_host(clusters, groups):
    for group in groups:
        for cluster_name in clusters:
            samogon_cluster = updater().clusters[cluster_name]
            if group in samogon_cluster.groups:
                return samogon_cluster, group
    return None, None


def _calc_clusters(mem_left, clusters, host_groups, host, use_gencfg_ip):
    result = {}
    for cluster in sorted_clusters(clusters):
        for group in host_groups:
            if group in cluster.groups:
                data, mem_left = place_cluster(cluster, group, host, mem_left, use_gencfg_ip)
                result.update(**data)
    return result, mem_left


def _calc_static(mem_left, clusters, host_groups, host, use_gencfg_ip):
    result = {}
    yt_cluster, group = _get_yt_on_host(clusters, host_groups)
    if yt_cluster:
        mem_left = yt_cluster.mem_min + max(0, mem_left)
        return place_cluster(yt_cluster, group, host, mem_left, use_gencfg_ip)
    return result, mem_left


def _calc_samogon(mem_left, clusters, host_groups, host, use_gencfg_ip):
    result = {}
    cluster, group = _get_samogon_on_host(clusters, host_groups)
    if cluster:
        mem_left = cluster.mem_min + max(0, mem_left)
        return place_cluster(cluster, group, host, mem_left, use_gencfg_ip)
    return result, mem_left


def is_satisfy(host, since, duration):
    # HACK: useful to smoothly change something in clusters mem/cpu requirements
    # or config format and so on.
    try:
        number = int(host.split('-')[1].split('.')[0])
    except (TypeError, IndexError, ValueError):
        return False
    prob = ((datetime.datetime.now() - since).total_seconds() / duration)
    if prob * 10000 > number:
        return True
    return False


def calc_clusters(
    resources, host,
    dynamic_clusters, dynamic_groups,
    static_clusters, static_groups,
    samogon_clusters, samogon_groups,
    use_gencfg_ip,
):
    result = {}
    mem_left = int(resources['mem']['free'])
    yt_cluster, yt_group = _get_yt_on_host(static_clusters, static_groups)

    if yt_cluster:
        mem_left -= MEM_AGENT_RESERVE
    if dynamic_clusters:
        mem_left -= MEM_AGENT_RESERVE
    if samogon_clusters:
        mem_left -= MEM_AGENT_RESERVE

    if (
        not yt_cluster
        or yt_cluster.mem_min + mem_left > MEM_YT_NEEDS + MEM_ROTOR_NEEDS
        or host.startswith('sas')
    ):
        # place rotor
        result, mem_left = _calc_clusters(
            mem_left,
            filter_clusters(dynamic_clusters, set(dynamic_clusters) & set(ROTOR_CLUSTERS)),
            dynamic_groups, host, use_gencfg_ip
        )
    # place yt
    static, mem_left = _calc_static(mem_left, static_clusters, static_groups, host, use_gencfg_ip)
    result.update(**static)

    # place samogon
    samogon, mem_left = _calc_samogon(mem_left, samogon_clusters, samogon_groups, host, use_gencfg_ip)
    result.update(**samogon)

    # place the rest
    dynamic, mem_left = _calc_clusters(
        mem_left,
        filter_clusters(dynamic_clusters, set(dynamic_clusters) - set(ROTOR_CLUSTERS)),
        dynamic_groups, host, use_gencfg_ip
    )
    result.update(**dynamic)

    return result


def host_port_to_dns_name(host, port):
    host = host.split('.')
    host[0] += '-{}.vm'.format(int(port))
    return '.'.join(host)
