import pymongo
import logging
import time
import sys

from libraries.topology.mongo import tags, searcher_lookup, groups
from libraries.utils import memoize


# grep stable- ~/work/svn_arcadia/gencfg/custom_generators/balancer_gencfg/src/mapping.py | cut -d '"' -f 12 | sort -u > /tmp/balancer_tags
#
# find /home/mcden/work/svn_nanny/ -type f -name '*.json' ! -exec grep -HL '"value": "OFFLINE"' {} \; |\
#     while read f; do grep '"release": "tags/stable-' $f; done | cut -d '"' -f 4 | cut -d / -f 2 | sort -u > /tmp/nanny_tags
#
# cat /tmp/balancer_tags /tmp/nanny_tags | sort -u

"""
heartbeat_mongodb_c:PRIMARY> db.tags.findOne({fullstate: 1})
{
    "_id" : ObjectId("57db04f730380470b206780f"),
    "tag" : "stable-92-r94",
    "commit" : "2469535",
    "fullstate" : 1
}
heartbeat_mongodb_c:PRIMARY> db.tags.findOne({diff_to: 1})
null
heartbeat_mongodb_c:PRIMARY> db.tags.findOne({diff_to: {$gt: 1}})
null
heartbeat_mongodb_c:PRIMARY> db.tags.findOne({diff_to: {$gt: "1"}})
{
    "_id" : ObjectId("5c402156d91c444dbc08fd85"),
    "tag" : "stable-125-r1506",
    "commit" : "4349635",
    "diff_to" : "4286602"
}
heartbeat_mongodb_c:PRIMARY> db.tags.findOne({commit: "4286602"})
{
    "_id" : ObjectId("5c11672d45251ab74625bdd1"),
    "tag" : "stable-125-r1",
    "commit" : "4286602",
    "fullstate" : 1
}
"""


CLEANUP_TO_TAG = 'stable-150-r1'


@memoize
def good_tags():
    good = set()
    for tag in tags().find({'fullstate': 1}):
        good.add(tag['tag'])
    return frozenset(good)


@memoize
def all_cleanup_tags():
    all_tags_ = set()
    for tag in tags().find({
        'commit': {
            '$lt': tag_to_commit()[CLEANUP_TO_TAG]
        }
    }):
        all_tags_.add(tag['tag'])
    return frozenset(all_tags_)


@memoize
def tag_to_commit():
    return {
        rec['tag']: rec['commit']
        for rec in tags().find()
    }


def get_cleanup_to_commit(tag):
    return tag_to_commit()[tag]


def tag_sorter(tag):
    _, major, minor = tag.split('-')
    return int(major) * 100000 + int(minor[1:])


def groups_remove_commit(commit):
    del_group_result_str = groups().delete_many(
        {
            'commit': str(commit)  # RIGHT TYPE!
        }
    )
    logging.warning('    %s removed from groups', del_group_result_str.deleted_count)
    return del_group_result_str.deleted_count


def searcher_lookup_remove_commit(commit):
    del_sl_result = searcher_lookup().delete_many(
        {
            'commit': int(commit)  # RIGHT TYPE!
        }
    )
    logging.warning('    %s removed from searcher_lookup', del_sl_result.deleted_count)
    return del_sl_result.deleted_count


def get_used_commits(used_tags, CLEANUP_TO_TAG):
    USED_COMMITS = set()

    for tag in tags().find({'tag': {'$in': list(used_tags) + [CLEANUP_TO_TAG]}}):
        USED_COMMITS.add(tag['commit'])
        if 'diff_to' in tag:
            USED_COMMITS.add(tag['diff_to'])

    return frozenset(USED_COMMITS)


def _get_unused_commits(collection, used_commits, upper_commit):
    return collection.distinct(
        'commit',
        {
            'commit': {
                '$nin': list(used_commits),
                '$lt': upper_commit
            }
        },
        projection={'_id': False, 'commit': True},
        sort=[('commit', pymongo.ASCENDING)],
    )


def get_unused_commits_from_groups(used_commits, upper_commit):
    return _get_unused_commits(groups(), map(str, used_commits), str(upper_commit))


def get_unused_commits_from_searcherlookup(used_commits, upper_commit):
    return _get_unused_commits(searcher_lookup(), map(int, used_commits), int(upper_commit))


def main():
    logging.warning('Read used tags from STDIN...')
    USED_TAGS = frozenset(tag.strip() for tag in sys.stdin)
    logging.warning('Have got %s used tags', len(USED_TAGS))

    assert len(USED_TAGS) > 100

    USED_COMMITS = get_used_commits(USED_TAGS, CLEANUP_TO_TAG)
    logging.warning('Have got %s commits in used tags', len(USED_COMMITS))

    CLEANUP_TO_COMMIT = get_cleanup_to_commit(CLEANUP_TO_TAG)
    logging.warning('Will not touch commit greater then %s', CLEANUP_TO_COMMIT)

    commits_to_remove = set(
        get_unused_commits_from_groups(USED_COMMITS, CLEANUP_TO_COMMIT) +
        get_unused_commits_from_searcherlookup(USED_COMMITS, CLEANUP_TO_COMMIT)
    )
    logging.warning('Going to remove %s commits', len(commits_to_remove))
    logging.warning('Wait 5 seconds before start...')
    time.sleep(5)
    for commit in sorted(commits_to_remove):
        logging.warning('Remove commit %s', commit)
        deleted1 = searcher_lookup_remove_commit(commit)
        deleted2 = groups_remove_commit(commit)

        if deleted1 + deleted2:
            sleep_time = 10
            logging.warning('        sleep %ss', sleep_time)
            time.sleep(sleep_time)


if __name__ == '__main__':
    main()
