#!/skynet/python/bin/python

"""
NB: Prepared to run with existing ./gencfgmain/ folder.
"""

import os, sys
sys.path.append(os.path.abspath(os.path.join(os.path.dirname(__file__), 'gencfgmain')))

import pymongo
import msgpack
import zlib
import bson
import datetime
import hashlib
import logging

from core.db import DB
from core.svnapi import SvnRepository

from mongo_params import INFRA_MONGODB

class SvnTopology(object):
    FIRST_OPERATIONAL_MAJOR_TAG = 58

    def __init__(self):
        self.repo = SvnRepository('./gencfgmain/')
        self.tags = self._parse_tags()

    def majors(self, reverse=False, highest=None, lowest=None):
        highest = highest or 2**31
        lowest = max(lowest, self.FIRST_OPERATIONAL_MAJOR_TAG)

        for tag in sorted(self.tags, reverse=reverse):
            if lowest <= tag <= highest:
                yield tag

    def minors(self, major):
        return sorted(self.tags[major])

    def get_groups(self, major, minor):
        self.repo.checkout('tags/stable-{}-r{}'.format(major, minor))

        db = DB('./gencfgmain/db/')
        try:
            return db.groups.get_groups()
        except (IOError, AttributeError) as e:
            print str(e)
            log('tag [{}] is unusable', self.repo.current_tag())
            return {}

    def get_commit_date(self):
        for line in self.repo.command(['info', '--xml']).stdout.splitlines():
            if line.startswith('<date>'):
                date = line.replace('<date>', '').replace('</date>', '').split('.')[0]
                return datetime.datetime.strptime(date, '%Y-%m-%dT%H:%M:%S')

    def _parse_tags(self):
        tags = {}

        for tag in self.repo.tags():
            try:
                if tag.startswith('stable-'):
                    major, minor = tag.split('-')[1:3]
                    major = int(major)
                    minor = int(minor.split('r')[1])

                    tags.setdefault(major, set())
                    tags[major].add(minor)

            except (IndexError, ValueError):
                pass

        return tags


class MongoTopology(object):
    uri = INFRA_MONGODB.uri

    def __init__(self):
        self.db = pymongo.MongoReplicaSetClient(
            INFRA_MONGODB.uri,
            connectTimeoutMS=500,
            replicaSet=INFRA_MONGODB.replicaset,
            w='majority',
            wtimeout=5000,
            read_preference=INFRA_MONGODB.read_preference,
        )['topology']

    def contains(self, major, minor):
        for _ in self.db.groups.find({'major': major, 'minor': minor}):
            return True
        return False

    def remove_version(self, major, minor):
        self.db.groups.remove({'major': major, 'minor': minor})

    def insert_group(self, name, major, minor, instances, master, extras=None):
        content = zlib.compress(msgpack.dumps(instances))

        doc = {
            'group': name,
            'version': major * 1000000 + minor,
            'major': major,
            'minor': minor,
            'time': datetime.datetime.utcnow(),
            'instances': bson.Binary(content),
            'md5hash': hashlib.md5(content).hexdigest(),
            'master': master,
        }
        doc.update(extras or {})

        self.db.groups.insert(doc)


def update(lowest, overwrite, dry_run):
    mt = MongoTopology()
    gt = SvnTopology()
    for major in gt.majors(lowest=lowest, reverse=False):
        for minor in gt.minors(major):
            try:
                if mt.contains(major, minor) and not overwrite:
                    log('version {}/{} already exists', major, minor)
                    continue

                if overwrite and not dry_run:
                    mt.remove_version(major, minor)
                if overwrite and dry_run:
                    log('removing {}/{}', major, minor)

                groups = gt.get_groups(major, minor)
                created = gt.get_commit_date()

                for group in groups:
                    try:
                        instances = list(group.get_kinda_busy_instances())
                        instance_records = [(inst.host.name, inst.port) for inst in instances]
                        cpu = long(sum(x.power for x in instances)) if not group.master else 0L
                        mem = long(group.reqs.instances.memory.value) * len(instances)

                    except KeyboardInterrupt:
                        return

                    except Exception as e:
                        log('group extraction failed: {}/{} {}', major, minor, group.name, e)
                        continue

                    master = group.master.name if group.master else None
                    if not dry_run:
                        mt.insert_group(
                            group.name,
                            major,
                            minor,
                            instance_records,
                            master, {
                                'cpu': cpu,
                                'mem': mem,
                                'count': len(instances),
                                'created': created,
                                'owners': group.owners,
                            }
                        )
                    else:
                        print instance_records
                        log('inserting group {} {}/{} master={}; extra={}', group.name, major, minor, master, {'cpu': cpu, 'mem': mem, 'count': len(instances), 'created': created})

                log('inserting {}/{} succeeded with {} groups', major, minor, len(groups))

            except KeyError:
                log('inserting {}/{} failed', major, minor)


def log(msg, *args):
    print str(datetime.datetime.now()), msg.format(*args)


def parse_args():
    from argparse import ArgumentParser
    parser = ArgumentParser(description = "Upload topology version to mongodb.")
    parser.add_argument('--overwrite', dest='overwrite', action='store_true', help='Overwrite existing group records.')
    parser.add_argument('--dry-run', dest='dry_run', action='store_true', help='Do not perform any real updates.')
    parser.add_argument('--lowest', dest='lowest', type=int, help='Lowest major tag')
    parser.set_defaults(overwrite=False)
    parser.set_defaults(dry_run=False)
    parser.set_defaults(lowest=0)

    return parser.parse_args()


def main():
    logging.basicConfig(level=logging.DEBUG)
    args = parse_args()
    update(args.lowest, overwrite=args.overwrite, dry_run=args.dry_run)


if __name__ == '__main__':
    main()
