import uuid
import random
from functools import wraps

import datetime
from api.logger import SkynetLoggingHandler
from kernel.util import logging
from kernel.util.functional import memoized, threadsafe, singleton


def run_daemon(target, args=()):
    from threading import Thread
    t = Thread(target=target, args=args)
    t.daemon = True
    t.start()


def shortname(host):
    return host.replace('.search.yandex.net', '').replace('.yandex.ru', '')


def logtime(fn):
    import time
    import math

    def exec_time(start_time):
        return str(math.trunc((time.time() - start_time) * 1000) / 1000.) + 's'

    @wraps(fn)
    def inner(*args, **kwargs):
        stime = time.time()
        retval = fn(*args, **kwargs)
        args[0].log.info('{0}: {1}'.format(fn.__name__, exec_time(stime)))
        return retval

    return inner


def run_daemon_loop(func, sleep_interval=60):
    def fn():
        import time

        while True:
            try:
                func()

            except Exception as e:
                logging.error(e)

            time.sleep(sleep_interval)

    run_daemon(fn)


def genuuid():
    return str(uuid.UUID(int=random.getrandbits(128)))


def timestamp(t):
    return (t - datetime.datetime(1970, 1, 1)).total_seconds()


@singleton
def configure_log(app='clusterstate', debug=False, beta=False):
    logger = logging.getLogger()
    handler = SkynetLoggingHandler(app=app, filename=(app if not beta else app + '-beta') + '.log')

    logging.initialize(logger=logger, level=logging.INFO, handler=handler, formatter=None)

    if debug:
        stdout_handler = logging.StreamHandler()
        stdout_handler.setLevel(logging.INFO)
        logging.initialize(logger=logger, level=logging.INFO, handler=stdout_handler)
    return handler


@singleton
def fqdn():
    import socket
    return socket.getfqdn()
