from __future__ import division, absolute_import

import urllib
import urllib2
import msgpack
import socket

from .connectorbase import ConnectorBase, ConnectorStatusResponse
from .utils import BoundHTTPHandler, resolve_all, DNSError
from .. import Task, Package, Service, Rules


class ConductorHTTP(ConnectorBase):
    def __init__(self, ctx):
        super(ConductorHTTP, self).__init__(ctx)
        self.cfg = self.ctx.cfg['connector']['conductorHTTP']
        self.log = self.ctx.log.getChild("ConductorHTTP")
        self.rules = Rules(self.ctx)
        self.apiUrl = self.cfg['apiUrl']
        self.url_opener = None

    def _http_request(self, url, **kwargs):
        if self.url_opener:
            result = self._http_request_perform(url, **kwargs)
            if result:
                return result

        fqdn = socket.getfqdn()
        addrs = resolve_all(fqdn)

        if not addrs:
            raise urllib2.URLError(RuntimeError("%s does not resolve to any ip address" % fqdn))

        addrs.sort(key=lambda a: (':' in a, a), reverse=True) # ipv6 first
        for addr in addrs:
            self.url_opener = urllib2.build_opener(BoundHTTPHandler((addr, 0)))
            self.url_opener.source_ip = addr

            result = self._http_request_perform(url, **kwargs)

            if result:
                return result

        raise urllib2.URLError(RuntimeError("Unable to communicate with conductor using any available address"))

    def _http_request_perform(self, url, **kwargs):
        try:
            return self.url_opener.open(url, **kwargs)
        except urllib2.URLError as error:
            self.log.error("Request from %s failed: %s", self.url_opener.source_ip, error)

    def refresh(self):
        """
        :rtype list[conductor.agent.task.Task]
        """
        url = "%s/tasks/%s?lock=true&format=msgpack&agent_version=%s&os_name=%s" % (
            self.apiUrl,
            socket.getfqdn(),
            urllib.quote_plus(self.ctx.version),
            urllib.quote_plus(self.ctx.system),
        )
        self.log.debug("Getting tasks from %s", url)
        try:
            result = self._http_request(url, timeout=self.cfg.get('connectionTimeout', 10))
        except urllib2.URLError as e:
            self.log.error("%s", e)
            return []
        except DNSError as e:
            self.log.error("%s", e.args[0])
            return []

        return self._createTasks(msgpack.loads(result.read()))

    def status(self, task, newStatus, log=None):
        """
        :type task: conductor.agent.task.Task
        :rtype: conductor.agent.connectors.connectorbase.ConnectorStatusResponse
        """
        error = None
        url = "%s/task_status?task_id=%d&status=%s&operation=%s&format=msgpack" % (
            self.apiUrl,
            task.id,
            newStatus,
            task.operation)
        request = urllib2.Request(url, log, {'Content-Type': 'text/plain'})
        try:
            result = self._http_request(request, timeout=self.cfg.get('connectionTimeout', 10))
        except urllib2.HTTPError as e:
            error = e
            self.log.error("Could not set status for task %s to %s: <Http status %d> %s" % (task.task, newStatus, e.getcode(), e.read()))
        except urllib2.URLError as e:
            error = e
            self.log.error("Could not set status for task %s to %s: %s" % (task.task, newStatus, e.reason))
        except DNSError as e:
            error = e
            self.log.error("%s", e.args[0])
        except socket.timeout as e:
            error = e
            self.log.error("Could not set status for task %s to %s: %s" % (task.task, newStatus, e.message))
        except Exception as e:
            error = e
            self.log.error("Unexpected exception while setting task %s status to %s: %s" % (task.task, newStatus, e.message))

        if error:
            return ConnectorStatusResponse(False, [])

        self.log.info("Status of task %s has been set to %s for operation %s", task.task, newStatus, task.operation)
        return ConnectorStatusResponse(True, self._createTasks(msgpack.loads(result.read())))

    def _createTasks(self, data):
        tasks = []
        for task_id in data:
            task_data = data[task_id]
            packages = task_data.pop('packages')
            removePackages = task_data.pop('remove_packages', {})
            packagelist = list(packages.keys()) + list(removePackages.keys())

            task = Task(
                task_id,
                task_data.pop('status'),  # Let's get possible error here, not deeper down
                task_data.pop('operation'),
                self.rules.get_schedule(packagelist),
                self.rules.get_parallel(packagelist),
                **task_data
            )

            for packageName in packages:
                package = self._makePackage(packageName, packages[packageName])
                task.addPackage(package)

            for packageName in removePackages:
                package = self._makePackage(packageName, removePackages[packageName])
                task.addPackage(package, remove=True)

            tasks.append(task)
        return sorted(tasks, key=lambda task: task.id)

    def _makePackage(self, name, data):
        services = data.pop('services')
        preServices = data.pop('preservices')

        package = Package(
            name=name,
            version=data.pop('version'),  # Let's get possible error here, not deeper down
            **data
        )

        for serviceName in services:
            service = self._makeService(package, serviceName, services[serviceName])
            package.addService(service)

        for serviceName in preServices:
            service = self._makeService(package, serviceName, preServices[serviceName])
            package.addService(service, pre=True)

        return package

    def _makeService(self, package, name, data):
        service = Service(
            name=name,
            package=package,
            **data
        )
        return service
