
from __future__ import division, absolute_import

import os
import subprocess
import time
import traceback

from ..log.file_proxy import FileProxy


class InstallerError(Exception):
    pass


class InstallerBase(object):
    def __init__(self, ctx):
        self.ctx = ctx
        self.cfg = self.loadConfig()
        self.log = self.getLog()
        self._task = None

    def doTask(self, task, logs):
        raise NotImplementedError

    def loadConfig(self):
        """
        :rtype: dict
        """
        raise NotImplementedError

    def getLog(self):
        """
        :rtype: logging.Logger
        """
        raise NotImplementedError

    @property
    def currentTask(self):
        """
        :rtype: conductor.agent.task.Task
        """
        return self._task

    def __get_output(self, title, logs):
        if self.ctx.logPath is None:
            return None

        base = os.path.abspath(self.ctx.logPath)
        task_dir = os.path.join(base, self._task.task)

        if os.path.exists(task_dir):
            if os.path.isdir(task_dir):
                if os.access(task_dir, os.W_OK):
                    pass
                else:
                    self.log.warning("No write access to %s. Cannot open stdout!" % task_dir)
                    return None
            else:
                self.log.warning("%s exists but is not a directory. Cannot open stdout!" % task_dir)
                return None
        else:
            try:
                os.mkdir(task_dir)
            except OSError:
                self.log.warning("Problem opening file for task logging", exc_info=True)
                file_io = open("/dev/null", mode="w")
                if logs:
                    logs.append("Problem opening file for task logging")
                    logs.append(traceback.format_exc())
                    return FileProxy(file_io, logs)
                return file_io
        output = os.path.join(task_dir, title)
        if os.path.exists(output):
            if os.path.isfile(output):
                if not os.access(output, os.W_OK):
                    self.log.warning("No write access to %s. Cannot open stdout!" % output)
                    return None
            else:
                self.log.warning("%s exists but is not a file. Cannot open stdout!" % output)
                return None
        try:
            file_io = open(output, mode="a")
        except IOError:
            self.log.warning("Problem opening file for task logging", exc_info=True)
            file_io = open("/dev/null", mode="w")
            if logs:
                logs.append("Problem opening file for task logging")
                logs.append(traceback.format_exc())
        if logs:
            return FileProxy(file_io, logs)
        return file_io

    def execute(self, command, shell=False, output=None, logs=None):
        msg = "Running %s" % command
        if logs:
            logs.append(msg)
        self.log.debug(msg)
        if output:
            f = self.__get_output(output, logs)
            if f:
                f.write("\n[%s] Running %s\n" % (time.asctime(), command))
                f.flush()
            try:
                proc = subprocess.Popen(command, shell=shell, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                                        preexec_fn=os.setsid)
                out = proc.communicate()[0]
                if f:
                    f.write(out)
                exit_code = proc.returncode
            finally:
                if f:
                    f.close()
            return exit_code
        else:
            return subprocess.call(command, shell=shell, preexec_fn=os.setsid)

    def executeWithOutput(self, command, shell=False):
        proc = subprocess.Popen(command, shell=shell, stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
                                preexec_fn=os.setsid)
        out = proc.communicate()[0]
        return proc.returncode, out
