from __future__ import division, absolute_import

import gevent

from .installerbase import InstallerBase, InstallerError
from .errors import IgnorableError


class LinuxInstallerError(InstallerError):
    pass


class PreServiceError(LinuxInstallerError):
    pass


class InstallError(LinuxInstallerError):
    pass


class RemoveError(LinuxInstallerError):
    pass


class ServiceError(LinuxInstallerError):
    pass


class LinuxInstaller(InstallerBase):
    SUDO_DEFAULT = True
    PRESERVICE_ACTION = 'restart'
    SERVICE_ACTION = 'restart'
    PACKAGE_DELIMITER = '='

    def __init__(self, ctx):
        super(LinuxInstaller, self).__init__(ctx)
        if 'sudo' in self.cfg:
            self.sudo = self.cfg['sudo']
        else:
            self.sudo = self.SUDO_DEFAULT
            self.log.info("Sudo flag set to default: %s" % self.SUDO_DEFAULT)

    def doTask(self, task, logs):
        sleep = getattr(task, 'sleep', 0)
        if sleep > 0:
            msg = "Sleeping %d seconds for task %s" % (sleep, task.task)
            logs.append(msg)
            self.log.info(msg)
            gevent.sleep(sleep)

        msg = "Executing operation %s for task %s: %s %s" % (
            task.operation,
            task.task,
            " ".join(["%s=%s" % (p.name, p.version) for p in task.packages]),
            " ".join(["%s-" % p.name for p in task.removePackages]),
        )
        logs.append(msg)
        self.log.info(msg)
        self._task = task
        self._logs = logs
        try:
            if task.operation == 'preservices':
                self.doPreServices(task)
            elif task.operation == 'install':
                self.doInstall(task)
            elif task.operation == 'services':
                self.doServices(task)
        finally:
            self._task = None
            self._logs = None

    def doPreServices(self, task):
        if not len(task.packages):
            return
        preServices = {}
        for package in task.packages:
            for preService in package.preServices:
                preServices[preService.name] = preService
        if not preServices:
            msg = "There's no preservices for task %s" % task.task
            self._logs.append(msg)
            self.log.debug(msg)
        else:
            msg = "Restarting preservices for task %s" % task.task
            self._logs.append(msg)
            self.log.info(msg)
            for service in preServices.values():
                command = service.script(self.PRESERVICE_ACTION, self.sudo)
                retCode = self.execute(command, shell=True, output="pre-service", logs=self._logs)
                if retCode:
                    raise PreServiceError("Command: %s returned %d" % (command, retCode))

    def doServices(self, task):
        if not len(task.packages):
            return
        services = {}
        for package in task.packages:
            for service in package.services:
                services[service.fullName()] = service
        if not services:
            msg = "There's no services for task %s" % task.task
            self._logs.append(msg)
            self.log.debug(msg)
        else:
            msg = "Restarting services for task %s" % task.task
            self._logs.append(msg)
            self.log.info(msg)
            for service in services.values():
                command = service.script(self.SERVICE_ACTION, self.sudo)
                retCode = self.execute(command, shell=True, output="service", logs=self._logs)
                if retCode:
                    raise ServiceError("Command: %s returned %d" % (command, retCode))

    def doUpdate(self, task):
        msg = "Reindex repos for task %s" % task.task
        self._logs.append(msg)
        self.log.info(msg)
        command = self.getUpdateCommand()
        retCode = self.execute(command, output="update", logs=self._logs)
        if retCode:
            msg = "Command %s returned %d" % (command, retCode)
            self._logs.append(msg)
            self.log.warn(msg)

    def doRemove(self, task):
        if not len(task.removePackages):
            return False
        msg = "Removing packages for task %s" % task.task
        self._logs.append(msg)
        self.log.info(msg)
        packageList = [package.name for package in task.removePackages]
        command = self.getRemoveCommand() + packageList
        retCode = self.execute(command, output="remove", logs=self._logs)
        for entry in self._logs:
            if "warning:" in entry:
                for s in entry.split("\n"):
                    if "warning:" in s:
                        self.log.warn(s)
        if retCode:
            msg = "Command %s returned %d" % (command, retCode)
            raise RemoveError(msg)

    def doInstall(self, task):
        self.doRemove(task)
        if not len(task.packages):
            return
        self.doUpdate(task)
        msg = "Installing packages for task %s" % task.task
        self._logs.append(msg)
        self.log.info(msg)
        packageList = ["%s%s%s" % (package.name, self.PACKAGE_DELIMITER, package.version) for package in task.packages]
        command = self.getInstallCommand(task)
        command += self.cfg.get('installerArgs', [])
        command += packageList
        retCode = self.execute(command, output="install", logs=self._logs)
        if retCode:
            msg = "Command %s returned %d" % (command, retCode)
            if self.commandIgnorable():
                msg += '. This attempt is ignored'
                raise IgnorableError(msg)
            raise InstallError(msg)

    def getInstallCommand(self, task):
        raise NotImplementedError

    def getRemoveCommand(self):
        raise NotImplementedError

    def commandIgnorable(self):
        return False

    def getUpdateCommand(self):
        raise NotImplementedError
