import os
import sys
import socket
import logging


try:
    from clint.textui import colored
except ImportError:
    # no colors in console
    colored = None


__all__ = ['setup_logging_to_stdout', 'setup_logging_to_file']


# Patch logging to have getChild function from Python 2.7.x
if not hasattr(logging.Logger, 'getChild'):
    def _getChild(self, suffix):
        if self.root is not self:
            suffix = '.'.join((self.name, suffix))
        return self.manager.getLogger(suffix)

    logging.Logger.getChild = _getChild


def _get_short_hostname():
    hostname = socket.gethostname()
    pos = hostname.find('.')
    if pos != -1:
        hostname = hostname[:pos]
    return hostname


class AugmentedFormatter(logging.Formatter):
    """
    Formatter that augments record instances with useful attributes
    """
    hostname = _get_short_hostname()

    def format(self, record):
        record.hostname = self.hostname
        return logging.Formatter.format(self, record)


if colored is not None:
    # colorful console is available
    # set appropriate formatter class
    class ColorfulFormatter(AugmentedFormatter):

        COLORS = {logging.ERROR: colored.red,
                  logging.WARN: colored.yellow}

        def format(self, record):
            color = self.COLORS.get(record.levelno)
            if color is not None:
                record.levelname = color(record.levelname)
            return AugmentedFormatter.format(self, record)

    Formatter = ColorfulFormatter
else:
    Formatter = AugmentedFormatter


class StreamToLogger(object):
    """
    Fake file-like stream object that redirects writes to a logger instance.
    From http://clck.ru/1DMQD.
    """
    def __init__(self, logger, log_level=logging.INFO):
        self.logger = logger
        self.log_level = log_level
        self.linebuf = ''

    def isatty(self):
        return False

    def write(self, buf):
        for line in buf.rstrip().splitlines():
            self.logger.log(self.log_level, line.rstrip())


# * prepend with host name
# * use asctime without backslashes to be able to grep without escaping
FORMAT = "%(hostname)s %(process)s %(asctime)s %(levelname)s [%(name)s] %(message)s"


def rename_loglevels():
    short_levelnames = {logging.WARN: "WARN ", logging.INFO: "INFO "}
    for k, v in short_levelnames.iteritems():
        logging.addLevelName(k, v)


def setup_logging_to_stdout():
    """
    :rtype: logging.Handler
    """
    rename_loglevels()
    formatter = Formatter(FORMAT)

    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    stdout_handler = logging.StreamHandler(sys.stdout)
    stdout_handler.setLevel(logging.DEBUG)
    stdout_handler.setFormatter(formatter)
    logger.addHandler(stdout_handler)

    return stdout_handler


def setup_logging_to_file(handler):
    """
    :param handler: log handler to configure
    :type handler: logging.handlers.Handler
    :return: configured logger
    :rtype: logging.Logger
    """
    rename_loglevels()
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    handler.setLevel(logging.DEBUG)

    formatter = Formatter(FORMAT)
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    # redirect standard streams to file too
    stdout_logger = logging.getLogger('STDOUT')
    stderr_logger = logging.getLogger('STDERR')

    # redirect stdout/stderr to /dev/null
    # all writes to them bypassing
    # sys.std[out|err] will be lost
    devnull = open(os.devnull, 'r')

    os.dup2(devnull.fileno(), 0)
    os.dup2(devnull.fileno(), 1)
    os.dup2(devnull.fileno(), 2)

    devnull.close()

    sys.stdout = StreamToLogger(stdout_logger, logging.INFO)
    sys.stderr = StreamToLogger(stderr_logger, logging.ERROR)

    return logger
